package Yandex::MyGoodWords;

use strict;
use Encode;
use Text::CLemmer2;
use List::MoreUtils qw/uniq/;

use Yandex::StopWords qw/is_stopword process_quoted_phrases/;

use utf8;

our $RUS_ALPHABET = 'абвгдеёжзийклмнопрстуфхцчшщъыьэюя';
our $MAX_WORDS_IN_KEYPHRASE ||= 7;

sub MyGetGoodWords {
    
    # $take_stopword - учитывать стоп слова(т.е. не пропукать их)
    my ($text, $take_stopword) = @_;
    $text = lc($text) if defined $text;

    # чтобы можно было использовать MyGetGoodWords в YT и чтобы у БК ничего не сломалось
    my ($profile);
    if (exists $INC{'Yandex/Trace.pm'}) {
        $profile = Yandex::Trace::new_profile('mygoodwords:MyGetGoodWords');
    }
    my $res = Text::CLemmer2::analyze($text, Text::CLemmer2::LEM_LANG_RUS);
    my (@lemmas, @formas);
    my %uniq;
    for my $word (@$res) {
        next if $word->{minus};

        next if !$take_stopword && is_stopword($word->{word});

        if ($word->{word} =~ /[$RUS_ALPHABET][a-z]|[a-z][$RUS_ALPHABET]/i) {
            # считаем несклоняемыми слова с буквами из разных языков
            # в БК реклама по таким фразам не показывается, поэтому 
            # это точка возможной накрутки ctr
            push @lemmas, lc($word->{word});
            push @formas, lc($word->{word});
            next;
        }

        for my $lemma (map {lc($_)} @{$word->{lemmas}}) {
            next if !$take_stopword && is_stopword($lemma) || $uniq{$lemma}++;
            push @lemmas, $lemma;
        }
        for my $forma (map {lc($_)} @{$word->{formas}}) {
            push @formas, $forma;
        }
    }
    return \@lemmas, \@formas;
}


#-------------------------------------
# Эта функция производит нормализацию
# фраз и возвращает обратно строку уже
# нормализованных слов через пробелы 
#-------------------------------------
#------------------------------------

sub norm_words
{
    my ($phrase, $NoWords, $mulform, $counter) = @_;

    # костыль для DIRECT-38795: Внести исключение для "купить" <> "куплю" в обнулении CTR
    my %norm_corrections = (
        'купля' => 'купить',
    );

    my $quoted_flag = 0;
    if ($phrase =~ /^\"[^\"]+\"$/) {
        $quoted_flag = 1;
        $phrase =~ s/^\"//;
        $phrase =~ s/\"$//;        
    }
    my $preserve_stopwords = $quoted_flag ? 1 : 0;
    $phrase =~ s/\[\s+/[/g;
    $phrase =~ s/\s+\]/]/g;
    $phrase =~ s/\]\[/] [/g;
    $phrase =~ s/([^\s])(\[)/$1 $2/g;
    $phrase =~ s/(\])([^\s])/$1 $2/g;

    my $twords = undef;
    my $gwords = _norm_words($phrase, $NoWords, $mulform, $counter, $preserve_stopwords);
    
    my @into_brackets;
    foreach my $arr (defined $gwords ? @$gwords : ()) {
        next unless $arr->[1] ne "" && $arr->[1] ne "-";
        
        
        if ($arr->[0] =~ /^\[[^\]]+\]$/) {
            push @{$twords}, '[' . $arr->[1] . ']';
        } elsif ($arr->[0] =~ /^\[/) {
            push @into_brackets, '[' . $arr->[1];  
        } elsif ($arr->[0] =~ /\]$/) {
            push @{$twords}, join ' ', @into_brackets, $arr->[1] . ']';
            @into_brackets = ();
        } else {
            if (@into_brackets) {
                push @into_brackets, split /\s+/, $arr->[1]
            } else {
                push @{$twords}, split /\s+/, $arr->[1];    
            }
        }
    }

    defined($twords) || return(undef);

    # выкидываем ключевики сверх лимита; минус-слова оставляем все
    my $num_kw = 0;
    @$twords = grep { /^-/ || ++$num_kw <= $MAX_WORDS_IN_KEYPHRASE } @$twords;

    my $word = join " ", sort &uniq(map {$norm_corrections{$_}//$_} @$twords);
    if ($quoted_flag) {
        $word = '"'.$word.'"';
    }

    return $word;
}

#-------------------------------------
# Эта функция производит нормализацию
# фраз и возвращает обратно строку уже
# нормализованных слов через пробелы 
#-------------------------------------
sub _norm_words
{
    my ($phrase, $NoWords, $mulform, $count, $preserve_stopwords) = @_;
    my $gwords = undef;
    $$count = 0;
    
    my ($into_bracket, $start_bracket, $end_bracket);
    # Создается цикл по словам во фразе
    foreach my $word (split(/\s+/, $phrase)) {

        # удаляем кавычки в начале и в конце слова
        $word =~ s/^"//;
        $word =~ s/"$//;

        my $w;
        # Проверяем на ключевые префиксы
        # Если они есть - не нормализуем
        if($word=~/^\d+\.\d+$/
        || $word=~/^\!/  || $word=~/^\~/ || $word=~/^\[\![^\]]+/
        || ($word=~/^\+/ && ($w = $word) =~ s/^\+// && is_stopword(lc($w)))
        || ($word=~/^-\!/ || $word=~/^-\~/) && defined $NoWords 
        || $preserve_stopwords && is_stopword(lc($word))
        ){  # Не надо обрабатывать те слова
            # которые имеют знаки '!' и '~'
            # (только приводим к нижнему регистру)

            $end_bracket = ($word =~ /\]$/);
            $into_bracket = 1 if $word =~ /^\[/;

            my $lc = lc($word);
            $lc =~ s#[\]\[]##g;
            push @{$gwords}, [$word, $lc];
        }
        else
        {   # В обратном случае нормализа
            # ция нужна и по каждому слову
            my $pref = $word =~ /^-/ ? '-' : '';
            next if !defined $NoWords && $pref eq '-';

            # Заменяем все небуквенно-цифровые
            # символы пробелами и делаем сплит
            $word =~s/[\-\+\~\!\&\%\$\^\_\"]/ /g;
            # случай число.число уже обработан в положительной ветке if
            $word = remove_dots($word) if $pref ne '-';
            $word =~ s/(?:^\s+|\s+$)//;

            $start_bracket = ($word =~ /^\[/);
            $end_bracket = ($word =~ /\]$/);
            $into_bracket = 1 if $start_bracket;

            $word =~ s#[\]\[]# #g;
            $word =~s/\s+/ /g;

            next if $word =~ /^\s*$/;

            my ($quantity, @normality) = _normalization(
                [grep {$_ ne ''} split /\s+/, $word], 
                $pref, $mulform,
                take_stopword => $into_bracket);
                
            if (@normality) {
                if ($start_bracket) {
                    $normality[0]->[0] = '[' . $normality[0]->[0];
                }
                if ($end_bracket) {
                    $normality[-1]->[0] = $normality[-1]->[0] . ']';
                }
            }
            push @{$gwords}, @normality;
            $$count += $quantity;
        }
    } continue {
        $into_bracket = 0 if $end_bracket;
    }

    # Возвратим рузультат
    return $gwords;
}

# Удаляет лишние точки и разбивает на подслова пробелами
# Логика разбивки как в CLemmer2
# R15.2 -> R 15.2
# 1.2.3.4.5 -> 1.2 3.4 5
# A.b -> A b
sub remove_dots {
    my ($word) = @_;
    my @matches = $word =~ /\d+\.\d+/g;
    my $result = '';
    foreach my $match (@matches) {
        my $match_index = index($word, $match);
        my $before_match = substr($word, 0, $match_index);
        $before_match =~ s/\.//g;
        $result .= ' ' if $result ne '';
        $before_match .= ' ' if $before_match ne '';
        $result .= $before_match . $match;
        $word = substr($word, $match_index + length $match);
    }
    $word =~ tr/./ /;
    return $result . $word;
}

sub _normalization {
    
    my ($words, $pref, $mulform, %options) = @_;
    $pref = '' unless defined $pref;

    my $cnt = 0;
    my @normality;
    foreach my $w (@$words) {
        my $nw = '';
        my $tw = $w;

        my ($tn, undef) = MyGetGoodWords($tw, $options{take_stopword});

        # Если получили нормализованное
        # слово, помещаем в массив фраз
        if (@$tn) {
            $cnt++;
            my @ws = @$tn;
            unless (defined $mulform) {
                $nw = (sort {
                    (length($a) <=> length($b))
                    || ($a cmp $b)
                } (@ws))[0];
            } else {
                @ws = sort {
                    (length($a) <=> length($b))
                    || ($a cmp $b)
                } (@ws);
                $#ws = 1 if ($#ws > 1);
                $nw = join(' ', @ws);
            }
        }
        push @normality, [$pref.$w, $pref.$nw];
    }
    
    return ($cnt, @normality);
}


sub get_whole_list_norm_words
{
    my $w = shift;
    my $norm_words = (MyGetGoodWords($w))[0];
    # Возвращаем undef для поддержания кода, который до изменений это функции ориентировался на undef.
    return (@$norm_words) ? $norm_words : undef;
}

=head2 get_lemmas_formas_list_words

    Берет из леммера все словоформы и леммы для фразы, сортирует и возвращает,

=cut
sub get_lemmas_formas_list_words
{
    my ($w, %options) = @_;

    # слова вида "<слово или цифра>-<цифра>" (например "ШПО-103") леммер сам разделяет на слова и минус-слова.
    # Поэтому, персонально для таких случаев разделяем фразу на два плюс-слова ("ШПО 103").
    $w =~ s/(?<!\s)-(\d+)/ $1/;
    my @lemmas_formas = MyGetGoodWords($w);
    push @{$lemmas_formas[1] || []}, split /\s+/, $w;
    foreach my $str (@lemmas_formas) {
        $str = [uniq @$str];
        $str = [sort @$str];
        $str = join ' ', @$str if $options{'str_result'};
    }
    return @lemmas_formas;
}


1;
