use strict;
use warnings;

=head1 DESCRIPTION

    Делает нужные use lib для Директовых модулей.

    Типичное использование:
        #!/usr/bin/perl
        use my_inc "..";
    
        #!/usr/bin/perl
        use my_inc "..", for => "protected";
    
        perl -e 'use my_inc for => "protected"'

    После этого не должны быть нужны никакие другие use lib

    Что происходит: 
    my_inc по относительному пути находит директорию layout/layout.yaml, смотрит, в каком файле находится вызывающий код, 
    сопоставляет с шаблонами и в соответствии с ними составляет список каталогов, которые надо использовать.

    TODO 
     * шаблоны и нужные подключаемые каталоги хранить отдельно, в файле про приложения и блоки

=cut

package my_inc;

use File::Basename qw/dirname/;
use Cwd qw/abs_path/;
use YAML::XS qw//;

use my_inc::Layout;

our @MY_INC;

my ($ROOT, $LAYOUT);
sub import 
{
    shift;
    my $rel_path = @_ && @_ % 2 ? shift : undef;
    my %O = @_;

    my $for_opt = delete $O{'for'};
    die "Unsupported options: ".join(',', keys %O) if keys %O;

    my $caller_filename = (caller())[1] // '.';
    my $caller_dir = abs_path(dirname($caller_filename));

    if (defined $rel_path && $rel_path =~ /^\//) {
        $ROOT = $rel_path;
    } elsif (defined $rel_path) {
        $ROOT = abs_path("$caller_dir/$rel_path");
    } else {
        $ROOT = abs_path(".");
    }

    _init_layout($ROOT) unless $LAYOUT;

    @MY_INC = dirs(defined $for_opt ? $for_opt : $caller_dir);

    my %inc_paths = map {$_ => undef} @INC;
    unshift @INC, grep {!exists $inc_paths{$_}} @MY_INC;
}


# внутренняя функция для инициализации $LAYOUT
sub _init_layout {
    my $file = "$ROOT/layout/layout.yaml";
    open(my $fh, "<", $file) || die "Can't open $file: $!";
    my $layout_yaml = do {local $/=undef; scalar <$fh>};
    close($fh) || die "Can't close $file: $!";

    $LAYOUT = YAML::XS::Load($layout_yaml);
    my_inc::Layout::normalize_blocks($LAYOUT->{blocks});    
}


=head2 my @dirs = my_inc::dirs(path)

    Получить список директорий, которые нужно использовать при компиляции модулей из path

=cut
sub dirs {
    my ($path) = @_;

    die "Layout not initialized" unless $LAYOUT;

    my ($result_block) = grep {$path =~ m!(?:^|.*/)\Q$_->{path}\E(?:/.*|$)!} @{$LAYOUT->{blocks}};
    die "can't find pattern for $path" unless $result_block;

    return map {"$ROOT/$_"} $result_block->{path}, @{$result_block->{includes}};
}


=head2 my $log = my_inc::path("../../log")

    аналог lib::abs::path - вычисление пути относительно файла, из которого делается вызов

=cut
sub path {
    (my $rel_path = shift) =~ s!^\./+!!;
    my $file = (caller())[1];
    (my $base_path = $file) =~ s![^/]+$!!;
    $base_path = abs_path($base_path || '.');
    return abs_path($base_path . '/' . $rel_path);
}


1;
