package my_inc::Layout;

use strict;
use warnings;

=head1 DESCRIPTION

Функции для работы с layout'ами проекта

Может использоваться: 
  * в my_inc, 
  * в юнит-тестах на консистентность layout'ов,
  * при разборе svn diff'ов по релизам

А отдельный юнит-тест на корректность layout'а можно не делать, 
если валидация делается каждый раз при загрузке.

=TODO

  - проверять описания блоков: есть все нужные поля, нет ненужных

=cut



=head2 normalize_blocks

Нормализует и проверяет layout:

  * разворачивает зависимости, 
  * проверяет существование включаемых блоков,
  * проверяет, что путь ни к какому блоку не является подкаталогом к другому блоку (атомарность использования блоков) -- подробнее см. validate_nesting_blocks

  На входе: 
    ссылка на массив описаний блоков

  Возвращаемое значение:
    нет 

  Модифицирует переданный массив

  Если проблемы -- умирает

=cut

sub normalize_blocks
{
    my ($blocks_layout) = @_;

    validate_nesting_blocks($blocks_layout);

    my %path_to_block = map { $_->{path} => $_ } @$blocks_layout;
    my %includes;
    for my $block ( @$blocks_layout ){
        my $block_path = $block->{path};
        for my $include_path ( @{$block->{includes}} ){
            $includes{$block_path}->{$include_path} = 1;
        }
    }

    my $something_changed = 1;
    my $iteration = 0;
    # цикл для разворачивания рекурсивных зависимостей
    while( $something_changed ){
        $iteration++;
        die "too deep recursion in block dependencies ($iteration)" if $iteration > 100;
        $something_changed = 0;
        # Проходим по всем блокам
        for my $block ( @$blocks_layout ){
            my $block_path = $block->{path};
            # Для каждого включаемого блока...
            for my $include_path ( keys %{$includes{$block_path}} ){
                # ... блок должен быть описан
                die "can't find block '$include_path', included in '$block_path'" unless exists $path_to_block{$include_path};
                # ... блок не должен включать сам себя -- ни прямо, ни опосредованно
                die "block $block_path tries to include itself" if $block_path eq $include_path;

                # проходим по зависимостям включаемого блока
                for my $recursive_include (@{$path_to_block{$include_path}->{includes}}){
                    # если наш блок уже использует этот -- ok, ничего больше не требуется
                    next if $includes{$block_path}->{$recursive_include};
                    # если еще нет -- записываем 
                    $includes{$block_path}->{$recursive_include} = 1;
                    $something_changed = 1;
                }
            }
        }
    }

    for my $block ( @$blocks_layout ){
        $block->{includes} = [sort keys %{$includes{$block->{path}}}];
    }


    return;
}


=head2 validate_nesting_blocks

  Проверяет, что блоки не вложены друг в друга в смысле каталогов. 

  На самом деле проверяет более строгое условие: 
  путь никакого блока не является префиксом пути никакого другого. 
  Это проверять проще.   

  На входе: 
    ссылка на массив описаний блоков

  Возвращаемое значение:
    нет 

  Переданный массив не модифицирет

  Если находит проблемы -- умирает

=cut

sub validate_nesting_blocks
{
    my ($blocks_layout) = @_;

    my @paths = sort map {$_->{path}} @$blocks_layout;

    for my $i ( 1 .. scalar(@paths) - 1 ){
        die "block '$paths[$i-1]' is a prefix of block '$paths[$i]'" if $paths[$i] =~ /^\Q$paths[$i-1]\E/;
    }

    return;
}


1;
