package Yandex::OAuth;

# $Id$

=head1 NAME

    Yandex::OAuth
    Модуль для работы с oauth.yandex.ru
    http://wiki.yandex-team.ru/oauth

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use JSON;
use LWP::UserAgent;
use HTTP::Headers;
use HTTP::Request;
use URI::Escape qw/uri_escape_utf8/;
use Data::Dumper;
use Time::HiRes;
use Encode;
use POSIX qw/strftime/;
use List::MoreUtils qw/uniq/;

use Yandex::Trace;
use Yandex::Log;
use Yandex::HTTP;

use utf8;

use base qw/Exporter/;
our @EXPORT = qw/
  oa_verify_token
  oa_verify_authorization
  oa_get_clients_apps
  oa_get_clients_own_apps
  oa_get_app_info
  oa_get_token_by_code
  /;

our $OAUTH_URL ||= 'https://oauth.yandex.ru/';
our $DEBUG = 0;
our @TIMEOUTS = (1, 4, 4);

our %OAUTH_LOG_SETTINGS;
%OAUTH_LOG_SETTINGS = (
    log_file_name => "oauth_calls.log",
    date_suf => "%Y%m%d",
) if !%OAUTH_LOG_SETTINGS;

=head2 $LOG_FILE
    
    если переменная LOG_FILE определена, файл открывается и в него пишется информация о ошибках

=cut
our $LOG_FILE;

=head2 $LOG_PROJECT
    
    записывается отдельным полем в лог

=cut
our $LOG_PROJECT;

=head2 oa_set_log($log_file, $log_project)

    установка лога

=cut
sub oa_set_log {
    ($LOG_FILE, $LOG_PROJECT) = @_;
}

=head2 oa_verify_token('vF9dft4qmT')

    проверка токена

=cut

sub oa_verify_token {
    my ($token) = @_;
    my $profile = Yandex::Trace::new_profile('oauth:verify_token');
    return oa_method('verify_token', { access_token => $token });
}

=head2 oa_verify_authorization(Authorization => Apache->request()->header_in('Authorization'))

    проверка всего заголовка авторизации

=cut
sub oa_verify_authorization {
    my %header = @_;
    
    my $profile = Yandex::Trace::new_profile('oauth:verify_authorization');
    return oa_method('verify_authorization', undef, { %header });
}

=head2 oa_get_clients_own_apps

    Пполучить приложения зарегистрированные клиентом

=cut

sub oa_get_clients_own_apps {
    my ($uid, $scope) = @_;
    $scope = [$scope] unless ref $scope eq 'ARRAY';
    
    my $profile = Yandex::Trace::new_profile('oauth:client:by_creator');
    my @result;
    foreach my $s (@$scope) {
        push @result, @{oa_method("client/by_creator/$uid", {scope => $s})};
    }
    @result = uniq @result;
    return \@result;
}

=head2 oa_get_clients_apps

    Получить приложения имеющие доступ к аккаунту криента

=cut

sub oa_get_clients_apps {
    my ($uid, $scope) = @_;
    my $profile = Yandex::Trace::new_profile('oauth:client:by_user');

    if (ref $scope eq 'ARRAY') {
        my @result;
        foreach my $s (@$scope) {
            push @result, @{oa_method("client/by_user/$uid", {scope => $s})};
            warn "oa_method client/by_user/$uid returned: " .Dumper \@result if $DEBUG;
        }
        @result = uniq @result;
        return \@result;
    } else {
        return oa_method("client/by_user/$uid", {scope => $scope});
    }
}


sub oa_get_app_info {
    my ($application_id) = @_;
    my $profile = Yandex::Trace::new_profile('oauth:client:info');

    my $app_info = oa_method("client/$application_id/info", {
        format => 'json',
        locale => 'ru'
    });
    warn "oa_method client/$application_id/info result: " . Dumper($app_info) if $DEBUG;

    return $app_info;
}

=head2 oa_get_token_by_code($params)

    получение токена по коду авторизации
        $params = {
            code => код авторизации ,
            client_id => '37375b50136245308d9af59fe1703c81',
            client_secret => 'd1932f07b36a484d86f6e18999d24f02'
        };

=cut

sub oa_get_token_by_code($)
{
    my $params = shift;
    my $profile = Yandex::Trace::new_profile('oauth:token');
    $params->{grant_type} = 'authorization_code';
    
    return oa_method("token", $params, ( request_method => 'POST' ));
}


# выполнение метода
sub oa_method {
    my ($url, $params, %O) = @_;

    my $log = new Yandex::Log(
        %OAUTH_LOG_SETTINGS,
        msg_prefix => "[$$]",
    );
    my $request_method = $O{request_method} || 'GET';
    my $header = $O{header} || {};
    $params ||= {};

    # сетапим свои заголовки если нет других
    $header->{'User-Agent'} ||= 'Yandex::OAuth/perl/1.00',
    $header->{'Accept'}     ||= 'application/json';

        # делаем http запрос
    my $ua = LWP::UserAgent->new(
                    default_headers => HTTP::Headers->new(%$header),
                );
    
    # создаём часть запроса с методом
    #my $url_part = serialize_params($params);
    my $query = serialize_params($params);
    
    my $req;

    if ($request_method eq 'POST'){
        # выполнение запроса
        $req = HTTP::Request->new($request_method => $OAUTH_URL . $url, undef, $query);
        print STDERR $req->as_string if $DEBUG;
    } elsif ($request_method eq 'GET') {
        my $url_part = "?$query" if $query;
        # выполнение запроса
        $req = HTTP::Request->new($request_method => $OAUTH_URL . $url . $url_part);
        print STDERR $req->as_string if $DEBUG;
    } else {
        die "Unknown method $request_method";
    }
    # пытаемся послать запрос несколько раз с разными таймаутами
    my $iter = 0;
    for my $timeout (@TIMEOUTS) {
        my $start_ts = Time::HiRes::time();
        $iter++;
        my $res;
        eval {
            $ua->timeout($timeout);
            my $resp = $ua->request($req);
            if (!$resp->is_success && $resp->code != 403) {
                die "OAuth request(timeout: $timeout, start: "
                    . format_ts($start_ts).')'
                    . ' request: ' . $req->as_string
                    . ' failed: ' . $resp->as_string;
            }

            print STDERR $resp->as_string if $DEBUG;

            my $content_type = $resp->header('Content-Type');
            if ($content_type eq 'application/json') {
                $res = eval {
                    JSON::from_json($resp->content, { utf8  => 1 });
                };
            } else {
                die "Unknown content type '$content_type'";
            }
            # перевариваем ответ
            if ($res && ref($res) eq 'HASH') {
                if ($res->{scope} && ref($res->{scope}) eq 'ARRAY') {
                    $res->{scope} = { map { $_ => 1 } @{$res->{scope}} };
                } elsif ($res->{scope}) {
                    $res->{scope} = { map { $_ => 1 } split(/\s+/, $res->{scope}) };
                }
            }
        };

        print STDERR Dumper($res) if $DEBUG;

        # проверяем успешность
        my $elapsed = Time::HiRes::time() - $start_ts;
        if (!$@) {
            log_access(0, $elapsed);
            return $res;
        }
        log_access(1, $elapsed);
        warn "OAUTH error: $@\n" if $DEBUG;
        # делаем паузу, если нужно
        if ($iter < @TIMEOUTS) {
            my $to_sleep = $timeout - $elapsed;
            Time::HiRes::sleep($to_sleep) if $to_sleep > 0;
        }
    }
    $log->out($@);
    die "OAUTH fatal error: $@";
}

sub serialize_params {
    my $params = shift || {};
    return join('&', (map {
        my ($k,$v) = ($_, uri_escape_utf8($params->{$_}) || '');
        if ($v && ref($v) eq 'ARRAY') {
            join('&', (map { "$k=$_" } @$v));
        } else {
            "$k=$v";
        }
    } keys %$params));
}

# форматирование timestamp для логов с милисекундами
sub format_ts {
    my $ts = shift;
    return strftime("%Y-%m-%d %H:%M:%S", localtime $ts).".".sprintf( "%03d", 1000*($ts-int($ts)) );
}

# вывод в лог информации о обращении к чя
sub log_access {
    my ($res_code, $answer_time) = @_;
    return if !$LOG_FILE;
    
    # TODO: переписать с использованием Yandex::Log
    open(my $log_fh, ">>", $LOG_FILE) || warn "Can't open '$LOG_FILE': $!";
    printf $log_fh "%d\t%s\t%d\t%d\n", time(), ($LOG_PROJECT||'-'), $res_code, 1000*$answer_time;
    close $log_fh;
}

1;
