package Yandex::Passport;

# $Id$

use strict;
use warnings;

use Carp;

use URI::Escape qw/uri_escape_utf8/;
use JSON;
use HTTP::Headers;

use Yandex::HTTP;
use Yandex::HashUtils;

use utf8;


=head2 $SERVICE_NAME_IN_PASSPORT

    Значение для поля from в запросе на создание логина
    Определяется в Settings.pm

=cut

our $SERVICE_NAME_IN_PASSPORT;

# https://wiki.yandex-team.ru/market/verstka/services/passport
our $PASSPORT_URL ||= "http://passport-internal.yandex.ru/passport";
our $PYTHON_PASSPORT_URL ||= "http://passport-internal.yandex.ru";

our $FAKE_IP ||= '127.0.0.1';
our $FAKE_USERAGENT ||= 'Direct fake user agent for server-side calls';


our @CALL_OPTIONS = qw/ http_method ip user_agent consumer tvm_ticket/;

our %PASSPORT_CALL = (
    create_track => {
        http_method => 'POST',
        url => '/1/track/',
        optional => [ qw/ track_type token language country display_language service / ],
    },
    validate_login => {
        url => '/1/validation/login/',
        required => [ qw/ login track_id / ],
        optional => [ qw/ ignore_stoplist / ],
    },
    suggest_login => {
        url => '/1/suggest/login/',
        optional => [ qw/ login firstname lastname language phone_number / ],
    },
    validate_password => {
        url => '/1/validation/password/',
        required => [ qw/ password / ],
        optional => [ qw/ login policy track_id / ],
    },
    create_passport_login => {
        http_method => 'POST',
        url => '/1/bundle/account/register/by_middleman/',
        required => [ qw/ login password firstname lastname language country / ],
        optional => [ qw/ track_id / ],
    },
);

=head2 unified_passport_call

    my $result = unified_passport_call( create_track => {display_language => "ru"} );

Запрос в Паспорт по одному из шаблонов, описанных в %PASSPORT_CALL.

Выполняет минимальную валидацию параметров, и формирует запрос в _python_passport_request

=cut

sub unified_passport_call {
    my ($call_id, $options, %O) = @_;

    my $call_info = $PASSPORT_CALL{$call_id};
    croak "Unknown passport call: $call_id"  if !$call_info;

    my $method = $call_info->{url};
    # todo: sprintf

    my $params = hash_cut $options, (@{$call_info->{required} || []}, @{$call_info->{optional} || []});
    for my $param ( @{$call_info->{required} || []} ) {
        croak "Required parameter is missed: $param"  if !defined $params->{$param};
    }

    my $call_opt = hash_cut \%O, @CALL_OPTIONS;
    $call_opt->{http_method} = $call_info->{http_method}  if $call_info->{http_method};

    return _python_passport_request($method, $params, %$call_opt);
}


=head2 _python_passport_request

    Делает запрос в HTTP-апи паспорта
    $response_struct = _python_passport_request($method, $request_params, ip => $ip, user_agent => $ua);

    https://wiki.yandex-team.ru/passport/python/api

=cut

sub _python_passport_request {
    my ($method, $params, %O) = @_;

    my $http_method = $O{http_method} ? $O{http_method} : 'POST';

    my $consumer = $O{consumer} || $SERVICE_NAME_IN_PASSPORT;
    my $url = Yandex::HTTP::make_url("$PYTHON_PASSPORT_URL$method", {consumer => $consumer});

    my $headers = HTTP::Headers->new('Ya-Consumer-Client-Ip' => $O{ip} || $FAKE_IP, 
                                     'Ya-Client-User-Agent' => $O{user_agent} || $FAKE_USERAGENT,
                                     ($O{tvm_ticket} ? ('X-Ya-Service-Ticket' => $O{tvm_ticket}) : ())
                                 );

    my $response = Yandex::HTTP::submit_form($http_method, $url, $params, default_headers => $headers);
    my $content = $response->content();
    if ($response->is_success()) {
        return decode_json($content);
    } else {
        my $status = $response->status_line();
        my $params_json = JSON->new->utf8->allow_nonref->encode($params);
        die "$status error accessing $url with $http_method params $params_json\nGot content in response:\n$content";
    }
}

=head2 create_passport_login

  fast create login in passport

    https://wiki.yandex-team.ru/passport/api/bundle/registration/#registracijaakkauntaposrednikom

  $result = create_passport_login($new_login, $name, $family, $password, ip => $ip, user_agent => $ua);

    Возвращает структуру 1-в-1 туже структуру, что получает от Паспорта.
    Описание полей смотри в паспортной вики-документации по ссылке выше.

=cut

sub create_passport_login
{
    my ($new_login, $name, $family, $password, %O) = @_;

    foreach my $param ( $new_login, $name, $family, $password ) {
        unless ( defined $param && length $param > 0 ) {
            die "not enough parameters given for create_passport_login";
        }
    }

    return _python_passport_request('/1/bundle/account/register/by_middleman/', {
        login => $new_login,
        password => $password,
        firstname => $name,
        lastname => $family,
        language => 'ru',
        country => 'RU',
    },
    %{hash_cut \%O, qw(ip user_agent tvm_ticket)});
}



=head2 get_login_suggestions

    Определение подходящих логинов

    https://wiki.yandex-team.ru/passport/python/api#opredeleniepodxodjashhixloginov

    $result = get_login_suggestions($login, firstname => $firstname, lastname => $lastname, language => $lang, phone_number => $phone);

    Возвращает структуру 1-в-1 туже структуру, что получает от Паспорта.
    Описание полей смотри в паспортной вики-документации по ссылке выше.

=cut

sub get_login_suggestions {
    my ($login, %O) = @_;
    
    my $tvm_ticket = delete $O{tvm_ticket};
    return unified_passport_call(suggest_login => {login => $login, %O}, 'tvm_ticket' => $tvm_ticket); 
}



=head2 subscribe_service

    подписываем паспортный логин на sid сервиса (по умолчанию на директовский: slova)

    $res = subscribe_service($uid) - подписывание, по умолчанию на 'slova'
    $res = subscribe_service($uid, 'strongpwd') - подписывание на другой sid
    $res = subscribe_service($uid, 'yastaff', 0, {yastaff_login => $domain_login}) - подписывание с дополнительными параметрами
    $res = subscribe_service($uid, 'strongpwd', 1) - отписывание
    
    $res == 0 - в случае успеха
    $res == 1 - если паспорт вернул ошибку (напрмер нет грантов)

=cut

sub subscribe_service($;$$$%) {
    my ($uid, $service_slug, $unsubscribe, $params, %O) = @_;

    $service_slug = $SERVICE_NAME_IN_PASSPORT unless $service_slug;

    my $result = eval { _python_passport_request("/1/account/${uid}/subscription/${service_slug}/", $params, ($unsubscribe?(http_method=>'DELETE'):(), %O) )};

    if (defined $result->{status} && $result->{status} eq 'ok') {
        return 0;
    } else {
        use Data::Dumper;
        my $error_message = $@ // 'unknown';
        warn "Error in sub subscribe_service!\nRequest: $uid, $service_slug, ".($unsubscribe?$unsubscribe:'').", ".Dumper($params)."\nResponse: ".Dumper($result) . "\nError: " . $error_message;
        return 1;
    }
}

1;
