package Yandex::Profile;

# $Id$

=head1 NAME

    Yandex::Profile
    Профилирование программы

=head1 SYNOPSIS

    use Yandex::Profile;

    # запуск первого таймера
    my $st = script_timer('coll_action' => $reqid);
    eval {
        # запуск второго таймера
        my $t2 = timer(op => "get");
        my $t3 = ntimer(123, op => "324");
        ...
        $t3->add_obj_num(10);
        ...
        # при выходе из зоны видимости таймер останавливается и собирается статистика
    };
    # если таймер нужно остановить раньше - можно разрушить объект самостоятельно
    $st = undef;

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use base qw/Exporter/;
our @EXPORT = qw/script_timer timer ntimer/;

=head2 $Yandex::Profile::LOG_FILENAME (default 'profile.log')

    Имя файла для выгрузки профайла исполнения (при использовании script_timer).
    Передаётся в Yandex::Log.

=cut
our $LOG_FILENAME ||= 'profile.log';

=head2 $Yandex::Profile::LOG_DATE_SUF (default '%Y%m%d')

    Суффикс логфайла (при использовании script_timer).
    Передаётся в Yandex::Log.

=cut
our $LOG_DATE_SUF;

=head2 $Yandex::Profile::REST_BORDER (default 0.01, т.е. 1%)

    Если неучтено времени больше, чем REST_BORDER, в лог добавится ещё один таймер

=cut
our $REST_BORDER ||= 0.01;


=head2 $Yandex::Profile::REST_func (default 'rest')

    Название таймера для аггрегации неучтённого времени

=cut
our $REST_FUNC ||= 'rest';

our %stats;

=head2 Yandex::Profile::init();

    Сбросить накопленную статистику.
    Нужно вызывать каждый раз при начале работы.

=cut
sub init {
    %stats = ();
}

=head2 my $t = timer(tag1 => val1, ...);

    Создать счётчик. При разрушении этого объекта в глобальный хэш добавится статистика.
    
    Примечание: На данный момент, теги и значения никак не разбираются, поэтому не стоит указывать
    в vals числовые значения, для этого используйте ntimer

=cut
sub timer {
    local $Yandex::Profile::Timer::caller_depth = $Yandex::Profile::Timer::caller_depth + 1;
    return Yandex::Profile::Timer->new(0, @_);
}

=head2 my $t = ntimer(obj_num, tag1 => val1, ...);

    Создать счётчик с указанием числа обрабатываемых "объектов". При разрушении этого объекта 
    в глобальный хэш добавится статистика.

=cut
sub ntimer {
    local $Yandex::Profile::Timer::caller_depth = $Yandex::Profile::Timer::caller_depth + 1;
    return Yandex::Profile::Timer->new(@_);
}

=head2 my $st = script_timer("bsClientData/$par_id" => $$);

    Создать "скриптовый" счётчик. При разрушении этого объекта глобальный хэш статистики сбросится в лог-файл.
    Сбрасывает накопленную статистику.

=cut
sub script_timer {
    return Yandex::Profile::ScriptTimer->new(@_);
}

=head2 my $reqid = current_reqid();

    Возвращает reqid последнего созданного (и пока неразрушенного) счётчика

=cut
sub current_reqid {
    if (my $st = $Yandex::Profile::ScriptTimer::_current_st_weakref) {
        return $st->{reqid};
    } else {
        return undef;
    }
}

=head2 Yandex::Profile::stats()

    Получить хэш со статистикой

=cut
sub stats {
    return \%stats;
}

=head2 print Yandex::Profile::stats_str(total=undef);

    Получить строку с профилем программы
    total - общее потраченное время, для генерации информации о неучтённом времени

=cut
sub stats_str {
    my ($total) = @_;

    my $rest_ela;
    if ($total && $total > 0) {
	my $sum_ela = 0;
	$sum_ela += $_->{ela} for values %stats;
	if (($total - $sum_ela) / $total > $REST_BORDER) {
	    $rest_ela = $total - $sum_ela;
	}
    }
    local $stats{"$REST_FUNC#"} = {ela => $rest_ela, cnt => 1} if $rest_ela;

    return join "\t",
        map {sprintf "%s: %.3f/%d/%d", $_, $stats{$_}->{ela}, $stats{$_}->{cnt}, $stats{$_}->{obj_num}||0}
        sort {$stats{$b}->{ela} <=> $stats{$a}->{ela}}
        keys %stats;
}


=head2 generate_reqid

    Сгенерировать уникальное 63-х битное целое неотрицательное числот - reqid

=cut
sub generate_reqid {
    return ( 
        (int(Time::HiRes::time()*256) & 0x7FFF_FFFF) << 32
        |
        ($$ & 0xFFFF) << 16
        |
        int(rand(0xFFFF))
        );
}


1;

package Yandex::Profile::Timer;

use strict;
use warnings;
use Time::HiRes;

our $caller_depth = 1;
our $stack_len = 1;

sub new {
    shift;
    my ($obj_num, %tags) = @_;

    # Кривое вычисление имени функции с пропуском eval-ов
    my $i = $caller_depth;
    my $subr;
    my @subrs;
    do {
        $subr = [caller($i++)]->[3];
        if ($subr && $subr ne '(eval)') {
            unshift @subrs, $subr;
        }
    } until !$subr || @subrs >= $stack_len;
    @subrs = ('unknown') if !@subrs;

    # формируем строку, по которой будет аггрегироваться статистика
    my $name = join('', map {"$_#"} @subrs).join("&", map {"$_=$tags{$_}"} sort keys %tags);
    bless {
        name => $name,
        obj_num => $obj_num,
        t1 => Time::HiRes::time(),
    };
}

=head2 $t->add_obj_num(10)

    Увеличить количество объектов на некоторую величину

=cut
sub add_obj_num {
    my ($self, $obj_num) = @_;
    $self->{obj_num} += $obj_num;    
}

sub DESTROY {
    my $self = shift;
    my $stat = $Yandex::Profile::stats{$self->{name}} ||= {};
    $stat->{ela} += Time::HiRes::time() - $self->{t1};
    $stat->{cnt} ++;
    $stat->{obj_num} += $self->{obj_num} || 0;
}

1;

package Yandex::Profile::ScriptTimer;

use strict;
use warnings;
use Time::HiRes;
use Scalar::Util qw/weaken/;

use Yandex::Log;
use Yandex::ProcInfo;

# слабая ссылка на последний созданный ScriptTimer
our $_current_st_weakref;

sub new {
    shift;
    my ($name, $reqid) = @_;

    Yandex::Profile::init();

    if (!defined $name) {
        if ($0 =~ /(\w+\.pl)$/) {
            $name = $1;
        } else {
            $name = '-';
        }
    }

    my $st = bless {
        name => $name,
        reqid => (defined $reqid ? $reqid : Yandex::Profile::generate_reqid()),
        t1 => Time::HiRes::time(),
        times1 => [times],
        mem1 => proc_memory(),
        pid => $$,
    };

    weaken($_current_st_weakref = $st);
    return $st;
}

=head2 my $name = $st->name(); $st->name($new_name);

    получение / установка названия счётчика после создания

=cut
sub name {
    my $self = shift;
    if (@_ == 0) {
        return $self->{name};
    } elsif (@_ == 1) {
        $self->{name} = shift;
    } else {
        die "Incorrect ScriptTimer->name() arguments";
    }
}

=head2 $st->profile_str()

    Получить отформатированную строку для лога

=cut
sub profile_str {
    my $self = shift;

    my @times = times;
    my $total = Time::HiRes::time() - $self->{t1};
    my $times_str = sprintf "%s:%.3f,%s:%.3f,%s:%.3f,%s:%.3f,%s:%.3f,%s:%d",
                        'total', $total,
                        'u', $times[0] - $self->{times1}->[0],
                        's', $times[1] - $self->{times1}->[1],
                        'cu', $times[2] - $self->{times1}->[2],
                        'cs', $times[3] - $self->{times1}->[3],
                        'mem', (proc_memory() - $self->{mem1}) / 1024 / 1024,
                        ;

    return join "\t", $self->{name}, $self->{reqid}, $times_str, Yandex::Profile::stats_str($total);
}

sub DESTROY {
    my $self = shift;

    # отслеживаем только то, что было создано в этом процессе
    if ($$ != $self->{pid}) {
        return;
    }

    my $profile_log = new Yandex::Log(
                        log_file_name => $Yandex::Profile::LOG_FILENAME,
                        date_suf => (defined $Yandex::Profile::LOG_DATE_SUF ? $Yandex::Profile::LOG_DATE_SUF : "%Y%m%d"),
                        );
    $profile_log->out($self->profile_str());
}

1;
