package Yandex::Redis::RateLimit;

=pod

    $Id$

=head1 NAME

    Yandex::Redis::RateLimit

=head1 SYNOPSIS
    
    my $rl = Yandex::Redis::RateLimit->new(time_interval => 100, max_requests => 3, name => 'api5-reports-requests-number');
    my $is_request_ok =$rl->request_ok( $self->subclient_client_id );
    unless ($is_request_ok) {
        return error_RateLimitExceeded(
            iget("Данный метод можно запрашивать не чаще чем %d раз в %d секунд", $rl->max_requests, $rl->time_interval)
        );
    }

=head1 DESCRIPTION

    Модуль для регулирования частоты обращений клиентов к сервисам Директа

    Поля:
    time_interval - интервал времени 
    max_requests - количество запросов которое можно сделать за time_interval
    name - префикс для ключа в Redis

    Методы:
    request_ok проверяет возможность сделать запрос в текущий интервал времени
        Если лимит не исчерпан и запрос сделать можно возвращает 1, если нет 0

=cut

use Direct::Modern;

use Mouse;

our $TIME_INTERVAL = 10;
our $MAX_REQUESTS = 13;

has redis           => ( is => 'ro', isa => 'Object', required => 1 );
has time_interval   => ( is => 'ro', isa => 'Int', default => sub { $TIME_INTERVAL; } );
has max_requests    => ( is => 'ro', isa => 'Int', default => sub { $MAX_REQUESTS; } );
has name            => ( is => 'ro', isa => 'Str', required => 1 );

=head2 request_ok($ident)

    Проверяет возможность сделать запрос в текущий интервал времени
    Если лимит не исчерпан и запрос сделать можно возвращает 1, если нет 0

=cut

sub request_ok {
    my ($self, $ident) = @_;
    my $dr = $self->redis();
    my $redis_key = $self->_get_key($ident);
    $dr->set($redis_key, 0, 'EX', $self->time_interval, 'NX', sub {});
    my $requests_number = $dr->incr($redis_key);
    if ( $requests_number && ( $requests_number > $self->max_requests ) ) {
        return 0;
    }
    return 1;
}

sub _get_key {
    my ($self, $ident) = @_;
    my $current_time = time();
    my $redis_key = $self->name . '-' . $ident . '-' . ( $current_time - ($current_time % $self->time_interval) );
    return $redis_key;
}

__PACKAGE__->meta->make_immutable();

1;
