package Yandex::ReportsXLS;

# $Id$

=head1 NAME

Yandex::ReportsXLS - удобное создание экселевских таблиц

=head1 DESCRIPTION

=cut

use Direct::Modern;

use IO::Scalar;
use Data::Dumper;
 
use Yandex::ReportsXLSWriter;
use Yandex::HashUtils qw/hash_cut/;

#-----------------------------------------------------------------------------------------------------------------------------

=head2 new(%options)

    Создание экземпляра Yandex::ReportsXLS

    Параметры:
        compatibility_mode => Режим улучшения совместимости со сторонними программами для чтения XLS.
                              Лучше использовать с осторожностью, т.к. для этого режима требуется дополнительная
                              память и дополнительные преобразования.

        use_old_merge_method => Использовать функцию объединения ячеек merge_cells вместо merge_range.
                                В этом режиме отключаются проверки на диапазоны мерджа и запись в ячейки происходит без
                                проверки на участие в мердже.

        no_optimization => Актуально для xlsx варианта модуля.
                            Отключает "оптимизацию" при формировании xlsx файла, которая существенно
                            уменьшает потребление памяти на больших объемах, но при этом перестает работать некоторая функциональность
                            (например не отрабатывает set_row)
                            http://search.cpan.org/~jmcnamara/Excel-Writer-XLSX-0.77/lib/Excel/Writer/XLSX.pm#SPEED_AND_MEMORY_USAGE
=cut

sub new{
    
    my $this = shift;
    my $class = ref($this) || $this;    
    my $self = {@_};    
    bless $self, $class;
    return $self;
}

sub init_xls_writer {
    my $self = shift;
    my $output = shift;
    return new Yandex::ReportsXLSWriter($output, %{ hash_cut($self, qw/compatibility_mode use_old_merge_method no_optimization/) });
}
#-----------------------------------------------------------------------------------------------------------------------------

=head2 array2excel2scalar

    return xls binary data

    my $xls = $xls_report->array2excel2scalar($xls_data, $xls_format);

=cut

sub array2excel2scalar{

    my $excel_data;
    my $self = shift;
    my $data = shift;
    my $format = shift;

    if (ref($format) ne 'ARRAY') {
        $format ||= {};
        $data = [$data];
        $format = [$format];
    }

    my $fh = new IO::Scalar \$excel_data;
    $format->[0]->{output} = $fh;
    $self->array2excel($data, $format);
    return $excel_data;
}

#-----------------------------------------------------------------------------------------------------------------------------

=head2 array2excel

 Непосредственно вывод в XLS, принимает 2 параметра - массив данных и хеш параметров/форматов

 массив данных это ссылка на 2-мерный массив, каждый элемент которого:
   - скаляр: выводятся просто как текст
   - undef: пустая ячейка (write_blank())
   - скаляр =~ /^=/: формула (write_formula())
     в формулах допустимо испльзовать вместо адресов ячеек имена ранее сохраненных ячеек.
     например: "={my_cell_name} + 123" вместо "=B1 + 123"

   - ссылка на хеш с ключами:
     data              => содержимое ячейки (см. выше)
     format            => ссылка на хеш для параметров форматирования ячейки,
                          такие же как в Spreadsheet::WriteExcel::add_format(),
                          добавляются к текущему global_format
     global_format     => тоже самое только действует на все последующие ячейки,
                          undef - отменяет текущий global_format, заданный ранее.
     add_global_format => добовляет форматы к текущему глобальному
     save_cell         => сохраняет адрес этой ячейки под именем (для использования в формулах)
     save_col          => сохраняет номер колонки под именем (для использования в $xls_format вместо абсолютного номера)
     save_row          => сохраняет номер строки под именем (для использования в $xls_format вместо абсолютного номера)
     url               => устанавливает гиперссылку на ячейку(допустимы ссылки на ячейки в файле (internal:) и внешние ссылки, например файлы(external:))
     comment           => устанавливает комментарий для ячейки
     as_text           => записать число в виде текста
     formula           => записываем формулу через write_formula()
     chart             => добавление диаграмма на страницу. Значение - ссылка на хеш с параметрами:
        type - тип диаграммы (см. Spreadsheet::WriteExcel::Chart)
        series - данные для построения диаграммы
        имя_метода => $params - вызов метода Spreadsheet::WriteExcel::Chart->set_"имя_метода"(%$params)

   my $xls_data =  [
            ["Текст ячейка 1", "Текст ячейка 2"],
            ["Текст ячейка 1", "Текст ячейка 2"],
            ["Текст ячейка 1", {data => "Текст ячейка 2", save_cell => 'my_cell_name'}],
            [{data => "Текст ячейка 1", format => {left => 2}, save_col => 'col_all'}, "Текст ячейка 2"]
   ];

 ссылка на хеш параметров/форматов с ключами:
    sheetname     => имя страницы в xls (по умолчанию "Report")
    sheetcolor    => цвет страницы в xls
    output        => file handle для вывода (по умолчанию STDOUT)
    set_row       => задать высоту или параметры группировки/сворачивания строк, ссылка на массив хешей с ключами:
      row    => номер строки (можно испоьзовать ранее сохраненное имя)
      height => высота
      hidden => см. документацию
      level => уровень группировки строк (см. документацию)
      collapsed => указывает что в данной строке будет +/- для разворачивания/сворачивания группы строк (см. документацию)
    set_column    => задать ширину колонок, ссылка на массив хешей с ключами:
      col1 => номер начальной колонки (можно испоьзовать ранее сохраненное имя)
      col2 => номер конечной колонки (можно испоьзовать ранее сохраненное имя)
      count => кол-во колонок (использовать вместо ключа col2)
      width => ширина
    merge_cells   => объединить ячейки, ссылка на массив хешей с ключами:
      row1      => номер 1-ой строки (допустимо имя)
      row2      => номер последней строки (допустимо имя)
      col1      => номер 1-ой колонки (допустимо имя)
      col2      => номер последней колонки (допустимо имя)
      col_count => кол-во колонок для объединения (вместо номера последней col2)
      row_count => кол-во строк для объединения (вместо номера последней row2)
    freeze_panes  => заморозить заданные ячейки (ссылка на массив)
    print_area  => установить область печати (ссылка на хэш с ключами общего формата)
    print_orient  => ориентация xls страницы при печати (скаляр - landscape|portrait)
    set_color => добавить пользовательские цвета (ссылка на массив массивов формата [newindex, R, G, B])

   my $xls_format = {
       sheetname => 'my sheet name',
       sheetcolor => 53,
       freeze_panes => [2, 2],
       set_row => [
           {row => 0, height => 15},
           {row => 1, height => 70}
       ],
       set_column => [
           {col1 => 0, col2 => 0, width => 15},
           {col1 => 1, count => 3, width => 20}
           {col1 => 4, col2 => 'last_column', width => 14},
       ],
       merge_cells => [
           {row1 => 0, row2 => 0, col1 => 0, col2 => 1},
           {row1 => 0, row_count => 2, col1 => 2, col_count => 3},
           {row1 => 0, row2 => 0, col1 => 'col_all', col_count => 5},
       ],
       print_area => {row1 => 0, row2 => 20, col1 => 1, col2 => 8},
       print_orient => 'landscape',
       set_color => [
           [40, 0xD2, 0xCD, 0xD9],
       ],
   };

   my $xls_report = ReportsXLS->new();      
   $xls_report->array2excel($xls_data, $xls_format);

=cut

sub array2excel{
    my $self = shift;
    my $data = shift;
    my $format = shift;

    if (ref($format) ne 'ARRAY') {
        $format ||= {};
        $data = [$data];
        $format = [$format];
    }

    my $output = $format->[0]->{output} || \*STDOUT;

    my $xls_writer = $self->init_xls_writer($output);

    # листы добавляем заранее, поскольку могут быть перекрестные ссылки на листы между собой, например за данными для диаграммы
    my @lists = ();
    for my $i (0 .. $#{$format}) {
        my $format = $format->[$i] || {};
        push @lists, $xls_writer->add_worksheet($format->{sheetname});
    }

    for my $i (0 .. $#{$format}) {
        my $list = $lists[$i];
        my $format = $format->[$i] || {};
        my $data = $data->[$i];

        $xls_writer->add_data($list, $format, $data);
    }
    
    $xls_writer->close();
}

=head2 _colnum2letter

=cut

sub _colnum2letter
{
    return Yandex::ReportsXLSWriter::_colnum2letter(@_);
}

1;
