package Yandex::ReportsXLSWriter;

# $Id: $

=head1 NAME

Yandex::ReportsXLSWriter - механизм для почанковой записи строк в xls

=head1 DESCRIPTION

=cut

use Direct::Modern;

use Spreadsheet::WriteExcel;

use IO::Scalar;
use Data::Dumper;
 
use Encode ();
use URI::Escape ();
use Yandex::IDN;

#-----------------------------------------------------------------------------------------------------------------------------

=head2 new(%options)

    Создание экземпляра Yandex::ReportsXLSWriter

    Параметры:
    
    $output - хендлер куда писать результат (обязательный)
    
    именованные параметры:
        compatibility_mode => Режим улучшения совместимости со сторонними программами для чтения XLS.
                              Лучше использовать с осторожностью, т.к. для этого режима требуется дополнительная
                              память и дополнительные преобразования.

        use_old_merge_method => Использовать функцию объединения ячеек merge_cells вместо merge_range.
                                В этом режиме отключаются проверки на диапазоны мерджа и запись в ячейки происходит без
                                проверки на участие в мердже.

        no_optimization => Актуально для xlsx варианта модуля.
                            Отключает "оптимизацию" при формировании xlsx файла, которая существенно
                            уменьшает потребление памяти на больших объемах, но при этом перестает работать некоторая функциональность
                            (например не отрабатывает set_row)
                            http://search.cpan.org/~jmcnamara/Excel-Writer-XLSX-0.77/lib/Excel/Writer/XLSX.pm#SPEED_AND_MEMORY_USAGE
=cut

sub new{
    
    my $this = shift;
    my $class = ref($this) || $this;    
    my $output = shift;
    die '$output handler must be defined' unless $output;

    my $self = {@_};    
    bless $self, $class;

    $self->{workbook} = $self->_init_xls_workbook($output);
    return $self;
}

sub normalize_format {
    my ($self, %format) = @_;
    return %format;
}

{
my %default_format = (
    size => 10,
    font => 'Arial', # установка шрифта не работает в Excel-Writer-XLSX-0.76
    align => 'left',
);
sub make_format {
    
    my ($self, %settings) = @_;
    my %format = $self->normalize_format(%default_format, %settings);
    return $self->{workbook}->add_format(%format);
}
}

sub _init_xls_workbook {
    my $self = shift;
    my $output = shift;
    my $xls = Spreadsheet::WriteExcel->new($output);
    $xls->compatibility_mode() if $self->{compatibility_mode};
    return $xls;
}

sub add_worksheet {
    my $self = shift;
    my $sheetname = shift;
    $self->{worksheets} //= [];
    my $worksheet_num = scalar(@{$self->{worksheets}});
    my $xls_worksheet = $self->{workbook}->add_worksheet($sheetname // 'Report' . ( $worksheet_num > 0 ? $worksheet_num : ''));
    my $worksheet = {xls_worksheet => $xls_worksheet,
                     row_num => 0};
    push @{$self->{worksheets}}, $worksheet;

    return $worksheet;
}

sub add_data {
    my ($self, $worksheet, $format, $data) = @_;

    my $list = $worksheet->{xls_worksheet};

    if ($format->{sheetcolor}) {
        $list->set_tab_color($format->{sheetcolor});
    }

    my %xls_formats_cache;
    my %xls_saved_named_cells;
    my (%rows_to_merge, %rowcol_to_merge_idx);

    if (ref($data) eq 'ARRAY' && @$data) {      

        my $row_num = $worksheet->{row_num} // 0;
        my @global_format;

        # resolve cells for formulas and formats
        for my $row (@$data) {
            next unless ref($row) eq 'ARRAY' && @$row;

            my $col_num = 0;
            for my $cell (@$row) {

                if (ref($cell) eq 'HASH') { # saved cells in hashrefs only

                    if ($cell->{save_cell}) {
                        $xls_saved_named_cells{ $cell->{save_cell} } = _colnum2letter($col_num) . ($row_num + 1);
                    }

                    if ($cell->{save_col}) {
                        $xls_saved_named_cells{ $cell->{save_col} } = $col_num;
                    }

                    if ($cell->{save_row}) {
                        $xls_saved_named_cells{ $cell->{save_row} } = $row_num;
                    }
                }

                $col_num++;
            }
            $row_num++;
        }
        
        if (ref($format) eq 'HASH' && $format->{merge_cells}) {
            foreach my $cell (@{$format->{merge_cells}}) {
                my @range = _get_range_info($cell, \%xls_saved_named_cells);
                push @{$rows_to_merge{$range[0]}}, {
                    range => \@range,
                    format => $cell->{format},
                };

                next if $self->{use_old_merge_method} || $self->{rewrite_merged_cells};

                my ($row_first, $col_first, $row_last, $col_last) = @range;
                next if $row_first == $row_last && $col_first == $col_last;
                for my $row ($row_first .. $row_last) {
                    for my $col ($col_first .. $col_last) {
                        $rowcol_to_merge_idx{"${row}:${col}"} = 1;
                    }
                }
            }
        }
        
        my %STORED_FORMULAS = ();
        # xls out
        $row_num = $worksheet->{row_num} // 0;
        for my $row (@$data) {
            next unless ref($row) eq 'ARRAY' && @$row;

            my $col_num = 0;
            my @columns;
            for my $cell (@$row) {

                my @cyr_format;
                my $cell_data;
                my ($url, $comment, $utf16be, $formula, $formula_value, $chart, $as_text);

                if (ref($cell) eq 'HASH') {
                    if (defined($cell->{url}) && $cell->{url} =~ m!^([a-z0-9]+://|internal:|external:)!i) {
                        $url = $cell->{url};
                    }

                    $chart = $cell->{chart} if $cell->{chart};

                    if (exists $cell->{comment}) {
                        $comment = $cell->{comment};
                    }

                    if (exists $cell->{utf16be}) {
                        $utf16be = $cell->{utf16be};
                    }

                    if (exists $cell->{as_text}) {
                        $as_text = $cell->{as_text};
                    }

                    if (exists $cell->{global_format}) {
                        @global_format = ref($cell->{global_format}) eq 'HASH' ? %{ $cell->{global_format} } : undef;
                    }

                    if (exists $cell->{add_global_format} && ref($cell->{add_global_format}) eq 'HASH') {
                        push @global_format, %{ $cell->{add_global_format} };
                    }
                
                    if (exists $cell->{image}) {                    
                        $list->insert_image($row_num, $col_num, $cell->{image}->{filename}, $cell->{image}->{x}, $cell->{image}->{y}, $cell->{image}->{scale_x}, $cell->{image}->{scale_y});                                        
                    }

                    if (exists $cell->{formula}) {
                        $formula = $cell->{formula};
                    }

                    if (exists $cell->{formula_value}) {
                        $formula_value = $cell->{formula_value};
                    }

                    push @cyr_format, %{ $cell->{format} } if $cell->{format};
                    $cell_data = $cell->{data};
                } else {
                    $cell_data = $cell;
                }
                
                push @cyr_format, @global_format;
                @cyr_format = grep {defined $_} @cyr_format;

                my $xls_format;
                if (@cyr_format) {
                    my $key = join '', sort @cyr_format;
                    if ($xls_formats_cache{$key}) {
                        $xls_format = $xls_formats_cache{$key};
                    } else {
                        $xls_format = $xls_formats_cache{$key} = $self->make_format(@cyr_format); 
                    }
                }
                
                push @columns, {data => $cell_data, format => {@cyr_format}};

                if ($rowcol_to_merge_idx{"${row_num}:${col_num}"}) {
                    # do not perform any writes in cells to merge -- this should be done in merge_range func
                    $col_num++;
                    next;
                }

                # write cell
                if ($formula || (defined $cell_data && !$as_text && $cell_data =~ /^=/)) {
                    $formula = $cell_data unless defined $formula; # для совместимости пока не перешли везде на ключ 'formula'
                    # is formula - resolve cell names for {cell_name}
                    $formula =~ s/\{(\w+)\}/_find_saved_num($1, \%xls_saved_named_cells, 'is_for_formula')/ge;
                
                    $self->_prepare_and_write_formula($list, $row_num, $col_num, $xls_format, $formula, $formula_value, $utf16be, \%STORED_FORMULAS);
                } elsif (defined $cell_data && $url) {
                    # decode utf16be if it
                    $cell_data = Encode::decode('utf16be', $cell_data) if $utf16be;
                    if ($url =~ /^(internal:|external:)/) {
                        $url =~ s/\{(\w+)\}/_find_saved_num($1, \%xls_saved_named_cells, 'is_for_formula')/ge;
                    } else {
                        my $ascii = Yandex::IDN::idn_to_ascii($url);
                        $url = $ascii if($ascii);
                        # escape url with utf symbols
                        $url = Encode::encode('utf8', $url, Encode::FB_DEFAULT()) if Encode::is_utf8($url, Encode::FB_DEFAULT());
                        $url = URI::Escape::uri_escape($url, "\x00-\x1f\x7f-\xff");
                    }                        
                    $list->write_url($row_num, $col_num, $url, $cell_data, $xls_format);
                } elsif (defined $cell_data) {
                    if ($utf16be) {
                        $list->write_utf16be_string($row_num, $col_num, $cell_data, $xls_format);
                    } else {
                        if ($as_text) {
                            $list->write_string($row_num, $col_num, $cell_data, $xls_format);
                        } else {
                            $list->write($row_num, $col_num, $cell_data, $xls_format);
                        }
                    }
                } elsif (defined $chart) {

                    my $graph = $self->{workbook}->add_chart(embedded => 1, type => delete $chart->{type});
                    my $code;
                    while (my ($function, $params) = each %$chart) {
                        $code->($graph, %$params) if ($code = $graph->can($function eq 'series' ? 'add_series' : "set_$function"))
                    }
                    $list->insert_chart($row_num, $col_num, $graph);
                } else {
                    $list->write_blank($row_num, $col_num, $xls_format);
                }

                if (defined $comment) {
                    $list->write_comment($row_num, $col_num, $comment);
                }

                $col_num++;
            }
            
            for my $cell ($rows_to_merge{$row_num} ? @{$rows_to_merge{$row_num}} : ()) {
                my @range = @{$cell->{range}}; 
                next if $range[0] == $range[2] && $range[1] == $range[3];
                my ($data, $format) = $columns[$range[1]] 
                    ? ($columns[$range[1]]->{data}, $columns[$range[1]]->{format}) 
                    : (undef, {});
                if ($self->{use_old_merge_method}) {
                    $list->merge_cells(@range);
                } else {
                    $list->merge_range(@range, $data,
                        $self->make_format($cell->{format} ? %{$cell->{format}} : %$format)
                    );
                }
            }
            $row_num++;
        }

        $worksheet->{row_num} = $row_num;
    }
    
    # apply formats
    if (ref($format) eq 'HASH' && %$format) {
        if ($format->{set_color}) {
            for my $cdata (@{$format->{set_color}}) {
                $self->{workbook}->set_custom_color(@$cdata);
            }
        }
        
        if ($format->{margins}) {
            my $i = 0;
            $list->can("set_margin_$_")->($list, $format->{margins}->[$i++]) foreach qw/left right top bottom/;
        }
        

        if ($format->{set_row}) {
            for my $fdata (@{$format->{set_row}}) {
                $list->set_row(_find_saved_num($fdata->{row}, \%xls_saved_named_cells), $fdata->{height}, undef, $fdata->{hidden}, $fdata->{level}, $fdata->{collapsed});
            }
        }
    
        if (defined $format->{hide_gridlines}) {
            $list->hide_gridlines($format->{hide_gridlines});            
        }
    
        if ($format->{freeze_panes}) {
            $list->freeze_panes(@{$format->{freeze_panes}});
        }

        if ($format->{set_column}) {
            for my $fdata (@{$format->{set_column}}) {
                my $col1 = _find_saved_num($fdata->{col1}, \%xls_saved_named_cells);
                my $col2;
                if (defined $fdata->{count}) {
                    $col2 = $col1 + $fdata->{count};
                } else {
                    $col2 = _find_saved_num($fdata->{col2}, \%xls_saved_named_cells);
                }
                $list->set_column($col1, $col2, $fdata->{width});
            }
        }

        if ($format->{print_area}) {
            my @range = _get_range_info($format->{print_area}, \%xls_saved_named_cells);
            $list->print_area(@range);
        }

        if ($format->{print_orient} && $format->{print_orient} =~ /^landscape|portrait$/) {
           if ($format->{print_orient} eq 'landscape') {
                $list->set_landscape();
           } elsif ($format->{print_orient} eq 'portrait') {
                $list->set_portrait();
           }
        }
    }
}

sub close {
    my $self = shift;

    if ($self->{workbook}) {
        $self->{workbook}->close();
        delete $self->{workbook};
    }
}

sub DESTROY {
    my $self = shift;

    $self->close();
}

#-----------------------------------------------------------------------------------------------------------------------------

sub _find_saved_num
{
    my ($value_or_key, $saved_values_hashref, $is_for_formula) = @_;

    die "dont defiend col or row num\n" unless defined $value_or_key;

    return $value_or_key if ($value_or_key =~ /^\d+$/ && ! $is_for_formula);

    my $result = $saved_values_hashref->{$value_or_key};
    die "dont find saved col/row num for \"$value_or_key\"\n" unless defined $result;

    if ($value_or_key =~ /^col_/ && $is_for_formula && $result =~ /^\d+$/) {
        $result = _colnum2letter($result);
    }

    return $result;

}

#-----------------------------------------------------------------------------------------------------------------------------

sub _get_range_info
{
    my ($fdata, $saved_values_hashref) = @_;

    my $col1 = _find_saved_num($fdata->{col1}, $saved_values_hashref);
    my $col2;
    if (defined $fdata->{col_count}) {
        $col2 = $col1 + $fdata->{col_count};
    } else {
        $col2 = _find_saved_num($fdata->{col2}, $saved_values_hashref);
    }

    my $row1 = _find_saved_num($fdata->{row1}, $saved_values_hashref);
    my $row2;
    if (defined $fdata->{row_count}) {
        $row2 = $row1 + $fdata->{row_count};
    } else {
        $row2 = _find_saved_num($fdata->{row2}, $saved_values_hashref);
    }
    return ($row1, $col1, $row2, $col2);
}

#-----------------------------------------------------------------------------------------------------------------------------

{
    my @all_letters = (("A" .. "Z"), ("AA" .. "ZZ"));

sub _colnum2letter
{
    my $num = shift;

    die "column number too long" if $num > $#all_letters;

    return $all_letters[$num];
}}

#-----------------------------------------------------------------------------------------------------------------------------

sub _prepare_and_write_formula 
{
    my ($self, $list, $row_num, $col_num, $xls_format, $formula, $value, $utf16be, $STORED_FORMULAS) = @_;
    $STORED_FORMULAS //= {};

    # для ускорения - преобразуем формулу в шаблон - заменяем переменные на A1 .. ANNN
    my $i = 1;
    my $formula_tmpl = $formula;
    my $replacements = {};
    $formula_tmpl =~ s/\b([A-Z]{1,3}\d{1,6})\b/'IV'.($replacements->{"IV$i"}=$1, $i++)/ge;

    $STORED_FORMULAS->{$formula_tmpl} ||= eval { $list->store_formula($formula_tmpl) };
    if ($STORED_FORMULAS->{$formula_tmpl}) {
        $list->repeat_formula($row_num, $col_num, $STORED_FORMULAS->{$formula_tmpl}, $xls_format, %$replacements);
    } else {
        # если не удалось записать формулу (например, она неверная или не формула вообще), пишем её как текст
        if ($utf16be) {
            $list->write_utf16be_string($row_num, $col_num, $formula, $xls_format);
        } else {
            $list->write_string($row_num, $col_num, $formula, $xls_format);
        }
    }
}

1;
