package Yandex::Retry;

=head1 NAME

    Yandex::Retry

    Сборище удобных попрограмм, контролирующих вызовы функций  

    Хочется переименовать во что-то вроде Yandex::ControledCall
    и добавить timeouted

=head1 SYNOPSIS

    # повторять операцию, пока она не выполнится, но не более 5-ти раз
    # ошибки выведутся в STDERR только если функция ни разу не выполнится удачно
    retry tries => 5, sub {
        do_something_dieable;
        open(...) || die;
        ...;
    };

    # при неудачах, вызвать функцию 5 раз - между первым и вторым вызовами - пауза 1 сек, между остальными - 2 сек.
    retry tries => 5, pauses => [1, 2], \&do_something_dieable;

    # после выполнения запроса в базу, спать в 2 раза больше времени
    relaxed times => 2, sub {do_sql(...)};

=cut

use strict;
use warnings;

use Time::HiRes qw//;
use Scalar::Util qw/looks_like_number/;

use Yandex::Trace;

use base qw/Exporter/;
our @EXPORT = qw/retry relaxed relaxed_guard/;

=head1 DESCRIPTION

=head2 $Yandex::Retry::iteration

    Переменная, содержащая порядковый номер итерации (начиная с 0)

=cut
our $iteration;

=head2 retry

    Функция принимает параметры и ссылку на функцию.
    Вызывает функцию несколько раз, пока она не выполнится.
    Возможные параметры:
        tries - число, максимальное количество вызовов, по умолчанию 2
        pauses - массив из положительных чисел, задержки между двумя соседними попытками, по умолчанию [0]
                если pauses слишком короткий - для последующих попыток используется последнее значение
    Возвращает результат функции

=cut
sub retry (@) {
    my $code = pop;
    # получаем параметры
    my $tries = 2;
    my @pauses = (0);
    while(@_) {
        my ($param, $val) = splice @_, 0, 2;
        if ($param eq 'tries') {
            if (defined $val && $val =~ /^\d+$/) {
                $tries = $val;
            } else {
                die "Incorrect tries param: '$val'";
            }
        } elsif ($param eq 'pauses') {
            if (ref($val) eq 'ARRAY') {
                @pauses = @$val;
            } else {
                die "Incorrect pauses param: '$val'";
            }
            
        } else {
            die "Incorrect param: '$param'";
        }
    }
    # цикл выполнения
    my @errors;
    my @ret;
    my $wantarray = wantarray;
    for $iteration (0..($tries-1)) {
        eval {
            if ($wantarray) {
                @ret = $code->();
            } else {
                $ret[0] = $code->();
            }
        };
        if ($@) {
            push @errors, "Try #$iteration: $@";
            my $pause = $pauses[$iteration <= $#pauses ? $iteration : $#pauses] || 0;
            if (@errors != $tries && $pause > 0) {
                my $profile = Yandex::Trace::new_profile('sleep:retry');
                Time::HiRes::sleep($pause);
            }
        } else {
            last;
        }
    }
    if (@errors == $tries) {
        die join "\n", @errors;
    }
    return wantarray ? @ret : $ret[0];
}


=head2 relaxed

    Функция принимает параметры и ссылку на функцию.
    Функция выплняется, после этого спим некоторое время 
    (чтобы не создавать лишнюю нагрузку на подсистемы)

    Возможные параметры:
        times - спим во столько раз больше, по сравнению с временем выполнения функции
                по умолчанию: 1

    Возвращает результат функции

=cut
sub relaxed (@) {
    my $code = pop;
    # получаем параметры
    my $times = 1;
    while(@_) {
        my ($param, $val) = splice @_, 0, 2;
        if ($param eq 'times') {
            if (defined $val && looks_like_number($val) && $val >= 0) {
                $times = $val;
            } else {
                die "Incorrect times param: '$val'";
            }
        } else {
            die "Incorrect param: '$param'";
        }
    }
    # цикл выполнения
    my $wantarray = wantarray;
    my $t1 = Time::HiRes::time();
    my @ret;
    if ($wantarray) {
        @ret = $code->();
    } else {
        $ret[0] = $code->();
    }

    my $profile = Yandex::Trace::new_profile('sleep:relaxed');
    Time::HiRes::sleep($times * (Time::HiRes::time() - $t1));

    return wantarray ? @ret : $ret[0];
}

=head2 relaxed_guard

    {
        my $guard = relaxed_guard times => 2;
        # .... do some computations ...
        # и на выходе из области видимости $guard проспит в 2 раза больше времени, чем было затрачено на вычисления.
    }

=cut
sub relaxed_guard(@) {
    Yandex::Retry::RelaxedGuard->new(@_);
}

package Yandex::Retry::RelaxedGuard;

sub new {
    shift;
    bless {
        times => 1,
        @_, # So we can override only 'times', but not 'start_time'
        start_time => [Time::HiRes::gettimeofday()],
    };
}

{
my $GLOBAL_DESTRUCTION = 0;

sub DESTROY {
    my $self = shift;
    return if $GLOBAL_DESTRUCTION;
    my $time_spent = Time::HiRes::tv_interval($self->{start_time});
    my $profile = Yandex::Trace::new_profile('sleep:relaxed');
    Time::HiRes::sleep($self->{times} * $time_spent);
}

sub END {
    $GLOBAL_DESTRUCTION = 1;
}
}

1;
