package Yandex::ScriptDistributor;

use Direct::Modern;

=pod

=encoding utf-8

=head1 NAME

Yandex::ScriptDistributor - сообщить скрипту, с какими параметрами он должен работать

=head1 DESCRIPTION

    Распределитель скриптов по шардам и специализациям.
    В кроне указывается только run_id скрипта: k (k-й запуск из N всего).
    В начале работы скрипт вызывает Yandex::ScriptDistributor::get_run_properties,
    и она определяет, с каким шардом работать.
    Можно сделать и определение специализации (heavy, std и т.п.).

=cut

use YAML;
use Yandex::Log::Messages;

use parent qw/Exporter/;

our @EXPORT = qw/
    get_run_properties
/;
our @EXPORT_OK = qw/
    juggler_vars
    get_run_count
    get_distribution
/;

=head2 $LOG

Можно проинициализировать логом скрипта, чтобы туда добавить данные о получении шарда и т.п.

=cut

our $LOG;

=head2 $USE_LOG_MESSAGES

Можно проинициализировать истинным значением, чтобы создавать объект Yandex::Log::Messages
и писать журнал в него, если явно не задан $LOG. См. описание конструктора этого класса для
требований работы с ним.

=cut

our $USE_LOG_MESSAGES;

our %CONF;

=head2 $SHARDS_NUM_REF

Предполагается изначально присвоить ссылку на число шардов в каком-то общем модуле, затем
можно переопределять число шардов не вспоминая, что в Yandex::ScriptDistributor тоже нужно

=cut

our $SHARDS_NUM_REF;
our $CONF_FILE_PATH;

=head2 _script_conf

Отдать конфигурацию для переданного имени скрипта; загрузить её, если ещё не загружена

=cut

sub _script_conf {
    my ($script_name) = @_;
    die 'Script name must be provided' unless $script_name;
    die 'Neither %CONF nor $CONF_FILE_PATH provided' unless (%CONF || defined $CONF_FILE_PATH);
    die "unknown number of shards" unless defined $SHARDS_NUM_REF && ref $SHARDS_NUM_REF;
    state $CONF;
    unless (defined $CONF) {
        $CONF = %CONF ? \%CONF : YAML::LoadFile($CONF_FILE_PATH);
    }

    $script_name = 'DEFAULT' unless exists $CONF->{$script_name};
    # если в конфигурации отсутствует DEFAULT
    die "unsupported script '$script_name'" unless exists $CONF->{$script_name};

    return $CONF->{$script_name};
}

=head2 get_run_properties

Получить ссылку на хеш с параметрами запуска скрипта

=cut

sub get_run_properties {
    my ($script_name, $run_id) = @_;

    my $C = _script_conf($script_name);
    die "run_id must be provided" unless $run_id;
    $run_id =~ m!^[1-9][0-9]*$! or die "run_id must be a positive integer; got '$run_id'";

    my $p;
    my ($count, $done) = (0, 0);
    for my $shard (1 .. $$SHARDS_NUM_REF) {
        my $shard_conf = (exists $C->{distribution}->{'shard_' . $shard}) ? $C->{distribution}->{'shard_' . $shard} : $C->{distribution}->{shard_default};
        die "no conf for shard $shard and no shard_default is defined" unless defined $shard_conf;
        for my $run_type (sort keys %$shard_conf) {
            $count += $shard_conf->{$run_type};
            if ($count >= $run_id) {
                $p->{shard} = $shard;
                $p->{run_type} = $run_type unless $run_type eq '_';
                $done = 1;
                last;
            }
        }

        last if $done;
    }

    die "run_id is too large, script $script_name, run_id $run_id" unless $done;

    $p->{uniq} = $run_id;

    if ($LOG || $USE_LOG_MESSAGES) {
        my $log;
        $log = $LOG ? $LOG : Yandex::Log::Messages->new();
        if (exists $p->{run_type}) {
            $log->out("Script $script_name with run_id $run_id got shard $p->{shard}, run_type $p->{run_type}");
        } else {
            $log->out("Script $script_name with run_id $run_id got shard $p->{shard}");
        }
    }
    return $p;
}

=head2 juggler_vars

Подготовить переменные для juggler-событий:
 - вернуть ссылку на хеш с ключами из возможных run_id, со значениями из ссылок на анонимные хеши
 - если в конфигурации есть специализации и ttl для них (juggler_ttl), то в значениях будет расставлены ttl

=cut

sub juggler_vars {
    my ($script_name) = @_;

    my $C = _script_conf($script_name);

    my $run_type_props = {};
    if (exists $C->{properties} && exists $C->{properties}->{run_type}) {
        $run_type_props = $C->{properties}->{run_type};
    }
    my ($vars_run_id, $count);

    for my $shard (1 .. $$SHARDS_NUM_REF) {
        my $shard_conf = (exists $C->{distribution}->{'shard_' . $shard}) ? $C->{distribution}->{'shard_' . $shard} : $C->{distribution}->{shard_default};
        die "no conf for shard $shard and no shard_default is defined" unless defined $shard_conf;
        for my $run_type (sort keys %$shard_conf) {
            if (exists $run_type_props->{$run_type}) {
                for (1 .. $shard_conf->{$run_type}) {
                    if (exists $run_type_props->{$run_type}->{juggler_ttl}) {
                        $vars_run_id->{++$count}->{ttl} = $run_type_props->{$run_type}->{juggler_ttl};
                    }
                }
            } else {
                $vars_run_id->{++$count} = {} for (1 .. $shard_conf->{$run_type});
            }
        }
    }

    return $vars_run_id;
}

=head2 get_run_count

Посчитать по имени скрипта, сколько нужно запустить его экземпляров, судя по конфигурации

=cut

sub get_run_count {
    my $vars = juggler_vars(@_);
    return scalar keys %$vars;
}

=head2 get_distribution

Для имени скрипта и количества запусков собрать хеш вида
(
    $shard => {
        $run_type => [
            $run_id,
            ...
        ],
        ...
    },
    ...
)

и отдать на него ссылку

=cut

sub get_distribution {
    my ($name, $num) = @_;

    die "script name must be provided" unless $name;
    die "number of instances must be provided and be a positive integer" unless $num && $num =~ m!^[1-9][0-9]*$!;

    my %distribution;
    for my $id (1 .. $num) {
        my $props = get_run_properties($name, $id);
        push @{$distribution{$props->{shard}}->{$props->{run_type} || '_'}}, $id;
    }

    return \%distribution;
}

1;
