# $Id$

=head1 NAME

    Yandex::SendSMS

=head1 DESCRIPTION

    Работа с SMS через Я.Паспорт

=cut

package Yandex::SendSMS;


use strict;
use warnings;
use utf8;

use Carp;

use POSIX qw(strftime);
use JSON;
use Path::Tiny;

use Yandex::HTTP;
use Yandex::SendMail;
use Yandex::Log;
use Yandex::Trace;
use Yandex::Blackbox;

our $PHONE_PASSPORT_URL = 'http://sms.passport.yandex.ru';


=head1 FUNCTIONS


=cut

use base qw/Exporter/;

our $VERSION = '0.01';

our @EXPORT = qw(
    send_single_sms
    sms_check_user
);

our @EXPORT_OK = qw(
);


=head1 $FAKESMS

    настройка перенаправления всех смс (для тестирования)
    
        $FAKESMS = "file:fakesms.log";        # в файл
        $FAKESMS = 'uid:xxx';                 # тестовому юзеру
        $FAKESMS = 'mailto:xxx@yyy.zz';       # на емейл
        $FAKESMS = 'null';                    # в никуда
        $FAKESMS = undef;                     # реальному пользователю

 
=cut

our $FAKESMS;

our $SMS_LOG = 'sms_send.log';


=head1 send_single_sms

    Отправляет смс юзеру по id

        my $result = send_single_sms( $uid, $text );

    Параметры:
        $uid  -- id пользователя
        $text -- текст сообщения

    Возвращаемое значение:
        1 -- ошибка
        0 -- OK    

    http://doc.yandex-team.ru/Passport/YaSMSDevGuide/reference/sendsms.xml

=cut

sub send_single_sms {
    my ( $uid, $sms_text, %O ) = @_;

    my $result;
    my ($proto, $addr) = ('uid', $uid);
    
    if ( defined $FAKESMS ) {
        ( $proto, $addr ) = $FAKESMS =~ m/^ (\w+) \s* (?: : \s* (.*) )?/x;
    }

    if ( $proto eq 'file' ) {
        # пишем в файл
        path($addr)->append_utf8(sprintf "%s: %s\n", strftime("%Y-%m-%d %H:%M:%S", localtime), $sms_text);
        $result = 'File';
    }
    elsif ( $proto eq 'uid' ) {
        # шлём смс через Паспорт
        my $sms_encoded = $sms_text;
        utf8::encode($sms_encoded);

        my %request_params = (
            uid => $uid,
            sender => 'direct',
            text => $sms_encoded,
            utf8 => 1,
        );
        my %options = (
            timeout => $O{timeout} // 5,
            ipv6_prefer => 1,
            handle_params => {keepalive => 1},
        );
        my $profile = Yandex::Trace::new_profile('sendsms:sendsms');
        $result = eval { Yandex::HTTP::http_fetch(GET => "$PHONE_PASSPORT_URL/sendsms", \%request_params, %options); };
        if (!defined $result) {
            carp "Can't sendsms: $@";
        }
    }
    elsif ( $proto eq 'mailto' ) {
        # отправляем на мейл
        send_alert( $sms_text, 'Redirected SMS message', $addr );
        $result = 'Mailed';
    }
    else {
        # ничего не делаем
        $result = 'Null';
    }

    # лог тоже ожидает данные в UTF-8
    my $log = Yandex::Log->new( log_file_name => $SMS_LOG, date_suf => '%Y%m' );
    $log->out({ dest => $uid, text => $sms_text, http_result => $result });
    
    return (! defined $result || $result =~ /<error>/);
}


=head1 sms_check_user

  check user in passport about phone for sms 

  my $phone = sms_check_user($uid); # undef or phone number

  http://doc.yandex-team.ru/Passport/YaSMSDevGuide/reference/userphones.xml

=cut

sub sms_check_user {
    my ($uid, $user_ip, %O) = @_;

    my $timeout = $O{timeout} // 5;
    
    my $number;
    eval {
        my $profile = Yandex::Trace::new_profile('sendsms:userphones');
        my $res = bb_userinfo({uid => $uid, getphone => 1}, $user_ip, 'direct.yandex.ru');
        $number = $res->{phone_number};
    };
    if ($@) {
        carp "Can't get userphones: $@";
        return undef;
    }

    return $number;
}


1;


=head1 AUTHORS

=cut


