package Yandex::Interactive;

=head1 NAME
    
    Yandex::Interactive
    $Id$
    Модуль, позволяет общаться с пользователем интерактивно в шеле

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use base qw/Exporter/;
our @EXPORT_OK = qw/
        prompt
        prompt_yn
        edit
        editor
        view
        viewer
/;

use Encode;
use File::Slurp;
use File::Temp qw//;
use POSIX qw//;
use Yandex::Shell;

use utf8;

=head2 prompt

    Спрашивает пользователя, проверяет ответ одним или несколькими правилами
    my $res = Yandex::Interactive::prompt("Edit? ", {"Edit? (Please enter y|e|q): " => qr/^(y|e|q)$/i});

=cut
sub prompt {
    my $msg = shift;
    my $checks = shift || {};
    print $msg;
    my $res;
  ITER:
    while($res = <STDIN>) {
        exit if !defined $res;
        $res =~ s/^\s+|\s+$//g;
        my $good = 1;
        for my $str (keys %$checks) {
            my $check = $checks->{$str};
            if (ref $check eq 'Regexp') {
                if ($res !~ $check) {
                    print "$str";
                    next ITER;
                }
            } elsif (ref $check eq 'ARRAY') {
                if (!grep {lc($_) eq lc($res)} @{$check}) {
                    print "$str";
                    next ITER;
                }
            } else {
                die "Unknown check type: '".ref($check)."'\n";
            } 
        }
        return $res;
    }
    exit;
}

=head2 prompt_yn

    Спрашивает вопрос, предполагающий ответ y/n
    exit if Yandex::Interactive::prompt_yn("Exit? ");

=cut
sub prompt_yn {
    return prompt($_[0], {"$_[0](Please enter y|n): " => qr/^(y|n)$/i}) =~ /y/i ? 1 : 0;
}

=head2 edit

    Позволяет пользователю редактировать текст
    my $msg = Yandex::Interactive::edit("template");

=cut
sub edit {
    my $msg = shift;
    my (undef, $file) = File::Temp::tempfile(UNLINK => 1);
    my $enc = console_encoding();
    write_file($file, Encode::is_utf8($msg) ? Encode::encode($enc, $msg) : $msg);
    eval {
        yash_system(editor(), $file);
    };
    if (!$@) {
        $msg = Encode::decode $enc, scalar read_file($file);
        chomp $msg;
    }
    return $msg;
}

=head2 editor

    Ищет предпочтительный редактор по стандартным правилам
    system(Yandex::Interactive::editor(), "file");

=cut
{
my $EDITOR;

sub editor {
    if (defined $_[0]) {
        return $EDITOR = $_[0];
    }
    if (!defined $EDITOR) {
        $EDITOR = '/usr/bin/editor';
        for my $e ($ENV{EDITOR}, $ENV{VISUAL}) {
            return $EDITOR = $e if defined $e && grep {-x "$_/$e"} split ':', $ENV{PATH}||'';
        }
    }
    return $EDITOR;
}
}

=head2 view

    Позволяет пользователю просмотреть текст
    Yandex::Interactive::view("text");

=cut
sub view {
    my $msg = shift;
    my (undef, $file) = File::Temp::tempfile(UNLINK => 1);
    my $enc = console_encoding();
    write_file($file, Encode::is_utf8($msg) ? Encode::encode($enc, $msg) : $msg);
    eval {
        yash_system(viewer(), $file);
    };
}

=head2 viewer

    Ищет предпочтительную программу для просмотра
    system(Yandex::Interactive::viewer(), "file");

=cut
{
my $VIEWER;

sub viewer {
    if (defined $_[0]) {
        return $VIEWER = $_[0];
    }
    if (!defined $VIEWER) {
        $VIEWER = '/usr/bin/pager';
        for my $e ($ENV{PAGER}) {
            return $VIEWER = $e if defined $e && grep {-x "$_/$e"} split ':', $ENV{PATH}||'';
        }
    }
    return $VIEWER;
}
}

# кодировка консоли
sub console_encoding {
    if (POSIX::setlocale(&POSIX::LC_CTYPE, "") =~ /\.([a-z0-9_\-]+)$/i) {
        my $decoder = Encode::find_encoding($1);
        if ($decoder && $decoder->perlio_ok) {
            return $decoder->name;
        }
    }
    return 'utf8';
}

1;
