#!/usr/bin/perl -w

=head1 SYNOPSYS

    Бенчмарк размера пачки спеллера

=head2 COMMENT

    От Дмитрия Сиваченко <mitya@yandex-team.ru>:

    Думаю, если задавать запросы в 10-50 потоков, то проблем для кластера быть не должно. И полагаю, что 50000 текстов проверятся очень быстро.

    Но и проверять, сколько максимально одновременных запросов выдержит кластер тоже не нужно (можно проверить на примере одной машины -- erratum-test -- и умножить на 25, именно столько машин в кластере).

    По результату тестирования на одной машине получилось, что максимальная выгода при запросе чанками по 400-500 байт
    и 10 одновременных потоках

=cut

use utf8;
use strict;
use warnings;

use Data::Dumper;
use FindBin qw( $Bin );
use lib "$Bin/../lib/";
use Yandex::Speller;
use Benchmark qw( timethis timestr );
use Test::Deep;

use constant TEST_COUNT => 6; # количество запусков по каждому тесту (для уточнения подсчёта времени)
use constant WORD_COUNT => 500; # общее количество слов в тексте
use constant ERR_WORD_COUNT => 5; # количество ошибочных слов (WORD_COUNT должно быть кратно ERR_WORD_COUNT)

use constant {
    WORD_CORRECT  => 'корова',
    WORD_INCORRECT  => 'карова',
};

my $expected_error = {
                        'error' => [
                                                 {
                                                     'len' => '6',
                                                     'word' => "\x{43a}\x{430}\x{440}\x{43e}\x{432}\x{430}",
                                                     'col' => '0',
                                                     'row' => '0',
                                                     'pos' => '0',
                                                     's' => [
                                                                        "\x{43a}\x{43e}\x{440}\x{43e}\x{432}\x{430}"
                                                                    ],
                                                     'code' => '1'
                                                 }
                                             ]
                    };

# приготовить тестовые данные
my (@bulk_input, @bulk_output);
foreach my $no (1 .. WORD_COUNT) {
    my $is_errorneous = (ERR_WORD_COUNT == 0) ? 0 : (($no % int(WORD_COUNT / ERR_WORD_COUNT)) == 0);
    if ($is_errorneous) {
        push @bulk_input, WORD_INCORRECT;
        push @bulk_output, $expected_error;
    } else {
        push @bulk_input, WORD_CORRECT;
        push @bulk_output, undef;
    }
}

$Yandex::Speller::SPELLER_TIMEOUT = 120;

my $test_results;

my @parallel_vars = (1..10);
my @size_vars = (100, 200, 300, 400, 500, 700, 900, 1000, 2000, 3000, 4000);

for my $parallel (@parallel_vars) {
    for my $size (@size_vars) {
        print "testing $parallel parallel requests, $size bytes\n";
        my $speller = Yandex::Speller->new(
                            parallel_reqs => $parallel, 
                            bulk_size => $size,
                            test_server => 1,
                        );
        my $params = { group => 1, safety => 1, skiplist => { 'yandex' => 1, 'Yandex' => 1 } };

        my $result;
        my $t = timethis(TEST_COUNT, sub { $result = $speller->check(\@bulk_input, $params) });

        my $time = timestr($t);
        if ($time =~ /\@\s+([.\d]+\/s)/) {
            $time = $1;
        }
        $test_results->{$parallel}{$size}{time} = $time;
        if ( cmp_deeply( $result, \@bulk_output, 'Bulk Test' ) ) {
            print "Test OK";
            $test_results->{$parallel}{$size}{result} = 'OK';
        } else {
            print "Test Fail";
            $test_results->{$parallel}{$size}{result} = 'Fail';
        }
    }
}

print "\n\nResult for ", WORD_COUNT, " words, ", TEST_COUNT, " repeats:\n\n";
print "threads\t\tsize\tresult\ttime\n";
foreach my $parallel (sort { $a <=> $b } keys %$test_results) {
    foreach my $size (keys %{ $test_results->{$parallel} }) {
        my $time = $test_results->{$parallel}{$size}{time};
        my $result = $test_results->{$parallel}{$size}{result};
        print "$parallel\t\t$size\t$result\t$time\n"; 
    }
}
