package Yandex::Speller;

use utf8;
use strict;
use warnings;
use Carp;

=pod
    
    $Id$
    Модуль реализующий API к сервису проверки опечаток
    Авторы:
        Брядов Василий <mirage@yandex-team.ru>
        Andrei Lukovenko <aluck@cordeo.ru>
    
=example
    
    Пример вызова:
        my $result = Yandex::Speller->new()->check("фраза с очепяткой", {group=>1, safety=>1, skiplist=>{'yandex'=>1,'Yandex'=>1}});

=options

    Конструктор класса:
    
        new Yandex::Speller
    
    Подробнее о параметрах: 
        https://wiki.yandex-team.ru/LingvisticheskieTexnologii/Opechatki/ServisOpechatok/SpellerApi/XMLSpellerApi

    По-умолчанию options = IGNORE_UPPERCASE + IGNORE_DIGITS + IGNORE_URLS + FIND_REPEAT_WORDS + IGNORE_LATIN

=cut

use XML::LibXML;
use Data::Dumper;
use Encode qw/encode_utf8/;
use HTTP::Request::Common;
use HTTP::Response;

use Yandex::HTTP qw( http_parallel_request );
use Yandex::Trace;

use constant {
    PRODUCTION_URL  => "http://speller.yandex.net/services/spellservice/checkTexts",
    TEST_URL        => "http://erratum-test.yandex.ru:19056/spellservice/checkTexts",
};

# для возможности ручного переопределения
our $SPELLER_URL ||= PRODUCTION_URL;
our $SPELLER_TIMEOUT ||= 1;

# макс. размер запроса - 10_000 байт (http://api.yandex.ru/speller/doc/dg/reference/checkTexts.xml)
# однако, Speller считает, что это с заголовком
use constant MAX_BULK_SIZE      => 8_000; 
use constant DEFAULT_BULK_SIZE  => 500;

# количество параллельных запросов
use constant {
    MAX_PARALLEL_REQS       => 10,
    DEFAULT_PARALLEL_REQS   => 6,
};

=head1 BENCHMARKS

    Результаты тестирования оптимальныъ размеров пакета:

        Пар. запросов       Размер запроса, байт
        1                   700
        2                   500
        3                   500
        4                   500
        5                   400
        6                   500
        7                   400
        8                   700
        9                   500
        10                  400

=cut

use constant {
    IGNORE_UPPERCASE            => 1, # пропускать слова, написанные заглавными буквами (ВПК)
    IGNORE_DIGITS               => 2, # пропускать слова с цифрами (авп17х4534)
    IGNORE_URLS                 => 4, # пропускать интернет-адреса, почтовые адреса и имена файлов
    FIND_REPEAT_WORDS           => 8, # подсвечивать повторы слов подряд (я полетел на на кипр)
    IGNORE_LATIN                => 16, # пропускать слова, написанные латиницей (madrid)
    NO_SUGGEST                  => 32, # только проверять текст, не выдавая вариантов для замены
    SPOPT_FLAG_LATIN            => 128, # отмечать как ошибочные слова, написанные латиницей
    SPOPT_BY_WORDS              => 256, # пословная проверка (считать соседние слова никак не связанными)
    SPOPT_IGNORE_CAPITALIZATION => 512, # игнорировать ошибки капитализации (неверное употребление прописных и строчных букв)
};

=head2 new

    Yandex::Speller->new(%params)

    $params{parallel_reqs} - количество одновременных запросов; по умолчанию Yandex::Speller::MAX_PARALLEL_REQS

    $params{bulk_size} - желаемый размер пакета, байт (по факту может отправиться пакет большего размера, если пришедший
        на проверку текст больше этого значения - текст не дробим)
    $Yandex::Speller::MAX_BULK_SIZE максимальное значение

    $params{test_server} - использовать тестовый сервер (по умолчанию - продакшн)

=cut

sub new 
{
    my $this = shift;
    my %params = @_;
    
    my $class = ref($this) || $this;
    my $self = {@_};
    bless $self, $class;

    $self->{parallel_reqs} = $params{parallel_reqs} || DEFAULT_PARALLEL_REQS;
    $self->{parallel_reqs} = MAX_PARALLEL_REQS if ($self->{parallel_reqs} > MAX_PARALLEL_REQS);

    $self->{bulk_size} = $params{bulk_size} || DEFAULT_BULK_SIZE;
    $self->{bulk_size} = MAX_BULK_SIZE if ($self->{bulk_size} > MAX_BULK_SIZE);

    $self->{server_url} = defined $params{test_server} ? TEST_URL : $SPELLER_URL;

    return $self;    
}

=head2 check

    $speller->check($data, $options)

    Если $data скаляр, возвращает одно значение с результатом (undef если $data не нуждается в исправлениях)

    Если $data arrayref, возвращает arrayref со значениями результатов в том же порядке

    NOTE: если нужна только проверка (есть ошибки/нет ошибок), то для ускорения ответа следует включать
    в $options флаг Yandex::Speller::NO_SUGGEST

=cut

sub check
{
    my ($self, $data, $options, $on_error) = @_;

    $on_error //= sub {};

    my $profile = Yandex::Trace::new_profile('speller:check');

    # создаем arrayref со входными значениями
    my $texts;
    if (!ref($data)) {
        return undef unless ($data//'') =~ /\S+/;
        $texts = [ $data ];
        $self->{raw_query} = $data;
    } elsif (ref($data) eq 'ARRAY') {
        $texts = $data;
    } else {
        carp "Unknown parameter type";
    }

    if (join('', map {$_//''} @$texts) !~ /\S+/) {
        return [map {undef} @$texts];
    }

    my $results = eval { $self->_request($texts, $options) };
    if ((!defined $results) || (scalar @$results == 0) || ($@)) {
        &$on_error();
    }
    if ($options->{safety}) {
        carp Dumper {SpellerError => $@} if $@;
    } else {
        croak Dumper {SpellerError => $@} if $@;
    }

    if (!ref($data)) {
        # сохраняем старое поведение
        $self->{raw_result} = $results->[0];
        return $self->{raw_result};
    } else {
        return $results;
    }
}

sub _request
{
    my ($self, $texts, $options) = @_;
    
    my %options = (
            ie => (defined $options->{ie} ? $options->{ie} : "UTF8"), 
            options => (defined $options->{options} 
                            ? $options->{options} 
                            : IGNORE_UPPERCASE + IGNORE_DIGITS + IGNORE_URLS + FIND_REPEAT_WORDS + IGNORE_LATIN
                        ), 
            lang => (defined $options->{lang} ? $options->{lang} : 'ru,en,uk'), 
        );
    my @options = map { $_, encode_utf8( $options{$_} )} keys %options;

    # формируем запросы
    my %requests;
    my $request;
    my $request_no = 0;
    my $processed = 0;
    my $text_size = scalar @$texts;
    while ($processed < $text_size) {
        my @params = @options;
        my $request_size;

        # формируем из @$texts пачку
        my $processed_last = $processed;
        do {
            push @params, 'text';
            push @params, $texts->[$processed++];
            $request = POST $self->{server_url}, \@params;  
            $request_size = $request->content_length();          
        } while (($processed < $text_size) && ($request_size < $self->{bulk_size}));

        # если очередной текст превысил максимальный размер пачки - урезаем его
        # исключение - если там и так всего один текст, тогда отсылаем пачку больше чем просил пользователь
        if (($request_size > $self->{bulk_size}) && ($processed - $processed_last > 1)) {
            pop @params; pop @params;
            $processed--;
            $request = POST $self->{server_url}, \@params;
        }

        $requests{$request_no++} = {
            method  => $request->method(),
            url     => $request->uri(),
            body    => \@params,
        };
    }

    return [] unless ($request_no);

    my %http_params = (
            headers => $request->headers(),
            timeout => ($options->{timeout} || $SPELLER_TIMEOUT),
            max_req  => $self->{parallel_reqs},
        );

    my $responses = http_parallel_request( 'POST', \%requests, %http_params ) || {};

    my $results;
    foreach my $id (sort { $a <=> $b } keys %{ $responses }) {
        my $response = $responses -> {$id};

        if ($response->{is_success}) {
            my $xml = $response->{content};

            push @$results, @{ _parse_spell_response($xml, $options) };
        } else {
            my $error_string = join( ' ', map { sprintf( '%s: %s.', $_, $response->{'headers'}{$_} ); } keys( $response->{'headers'} || {} ) );
            die( __PACKAGE__ . ' error: ' . $error_string );
        }
    }

    return $results;
}

sub _parse_spell_response
{
    my $text_xml = shift;
    my $options = shift;
    
    my $doc = XML::LibXML->new()->parse_string($text_xml);
    my $results;

    foreach my $spell_result ( $doc->getElementsByTagName('SpellResult') ) {
        my ($result, @errors) = ();    
        
        foreach my $node ( $spell_result->getElementsByTagName('error') ) {
            my $erdesc = {};
            if (my $word_node = ($node->getChildrenByTagName('word'))[0]) {
                $erdesc->{word} = $word_node->textContent || '';
            }
            
            next if defined $erdesc->{word} 
                        && defined $options->{skiplist} 
                        && defined $options->{skiplist}{$erdesc->{word}};
            
            map {$erdesc->{$_} = $node->getAttribute($_)} qw/code pos row col len/;
            
            if (my @suggest = $node->getElementsByTagName('s')) {
                foreach my $s (@suggest) {
                    push @{$erdesc->{s}}, $s->textContent || '';
                }
            }    
            
            push @errors, $erdesc;
        }
        
        if (@errors) {
            if ($options->{group}) {
                # т.к. ответ спеллера для неск-ких один фраз с ошибкой - содержит подсказки для каждой из ошибочных фраз
                # ответ группируем по фразам
                
                my $hresults = {};            
                foreach my $er (@errors) {
                    unless (defined $hresults->{$er->{word}}
                            && $hresults->{$er->{word}}) {
                        $hresults->{$er->{word}} = $er;
                    }
                }
                
                foreach my $word (keys %$hresults) {
                    push @{$result->{error}}, $hresults->{$word};
                }
                
            } else {
                $result->{error} = \@errors;
            }
        }

        push @$results, $result;
    }    
    
    return $results;
}

sub highlight
{
    my $self = shift;
    my $new_text = $self->{raw_query};
    
    my $pre = q{<font color="red">};
    my $post = q{</font>};
    my $errs = $self->{raw_result}->{error};
    
    return $new_text if !defined $errs;
    
    foreach my $l (@$errs) {
        if (defined $l->{word}) {
            $new_text =~ s/(\Q$l->{word}\E)/${pre}$1${post}/gsi;
        }
    }
    
    return $new_text;
}

1;
