package Yandex::Staff3;

# $HeadURL$
# $Id$

=head1 NAME

    Yandex::Staff3
    Perl обертка для API 3-ей версии к Стаффу: https://staff-api.yandex-team.ru/v3/persons

=head1 DESCRIPTION

    Для работы необходимо зарегистрированное приложение для доступа к OAuth с параметрами:
    our $STAFF_OAUTH_TOKEN;
    our $STAFF_OAUTH_URL; - это адрес API, не обязательный параметр, если не задан, то
        будет использовано значение: https://staff-api.yandex-team.ru/v3/persons
    Регистрация выполняется начиная со страницы: https://wiki.yandex-team.ru/intranet/dev/oauth

=cut

use base qw/Exporter/;
use strict;
use warnings;
use utf8;

use bytes               ();
use Data::Dumper        qw( Dumper );
use Carp                qw( cluck confess );

use JSON                ();
use HTTP::Headers       ();
use HTTP::Request       ();
use List::UtilsBy       qw( max_by );
use URI::Escape::XS     qw( uri_escape );
# https://st.yandex-team.ru/DIRECTADMIN-5729
#use IO::Socket::SSL     qw/ SSL_VERIFY_NONE /;

use Yandex::HTTP        ();
use Yandex::ListUtils   qw( chunks );

our @EXPORT = qw();
our @EXPORT_OK = qw(load_token_from_file get_staff_info get_mass_staff_info get_staff_info_ext);

our $STAFF_OAUTH_TOKEN;
our $STAFF_OAUTH_URL ||= 'https://staff-api.yandex-team.ru/v3/';

=head2 $STAFF_MAX_SIZE_URL & $STAFF_REQUEST_METHOD

    Ограничение на длину GET-запроса в API Стафф'а, так как было получены письма от Стаффа,
    он же Александр Кошелев: "Сейчас ограничение примерно в 4000 байт.",
    "На 4000 можешь рассчитывать.", и в итоге:
    "В конфиге действительно 4k лимит на буфер. Но для урла максимум 1350 cимволов получается."
    $STAFF_REQUEST_METHOD = 'GET' cейчас единственный вариант, поддерживаемый API Стаффа,
    добавлен на будущее, если они сделают таки задачу https://st.yandex-team.ru/STAFF-3455

=cut

our $STAFF_MAX_SIZE_URL   ||= 1350;
our $STAFF_REQUEST_METHOD ||= 'GET';

use constant LIMIT_PERSON_PER_PAGE => 50; # по сколько персон максимум забирать со Стаффа
use constant ESCAPED_COMMA_LENGTH  => length(uri_escape( ',' )); # длина запятой в URL
use constant DEFAULT_PATH_TO_TOKEN_FILE => "/etc/direct-tokens/startrek"; # по дефолту берем токен для стаффа отсюда


sub load_token_from_file {
    my $path_to_token = shift;
    $path_to_token ||= DEFAULT_PATH_TO_TOKEN_FILE;

    open(my $fh, "<", $path_to_token) || ($path_to_token eq DEFAULT_PATH_TO_TOKEN_FILE ? return : confess "Can't open $path_to_token: $!");
    my $token = <$fh>;
    $STAFF_OAUTH_TOKEN = $token;
    close($fh) || confess "Can't close $path_to_token: $!";
}


BEGIN {
    load_token_from_file();
}


=head2 get_staff_info($domain_login; \@fields)

    Получаем поля \@fields со стаффа по логину $domain_login.
    Возвращаем хеш с данными пользователя, в ключах которого имена полей.
    Если пользователя не нашлось, то вернётся пустой хеш.

=head2 SYNOPSIS

    Пример использования:
    my $staff = get_staff_info($domain_login, ['login', 'work_phone']);
    my ($login, $phone) = ($staff->{'login'}, $staff->{'work_phone'});

=cut

sub get_staff_info {
    my $domain_login = shift;
    return {} unless $domain_login;

    my $fields = shift || [qw(id uid guid login work_email work_phone)]; # остальные поля не просто строчки, а сложнее: https://staff-api.yandex-team.ru/_static/api/schemas/person.jso

    my $staff = get_mass_staff_info([$domain_login], $fields);

    if ( $staff && ref($staff) && exists($staff->{$domain_login}) && ref($staff->{$domain_login}) ) {
        return $staff->{$domain_login};
    }
    else {
        return {};
    }
}

=head2 get_mass_staff_info(\@domain_login; \@fields)

    Получаем поля \@fields со стаффа по логинам в \@domain_login.
    Возвращаем ссылку на хеш с ключами - логинами юзеров, по которым расположены 
    хеши с данными пользователя, в ключах которых имена полей.
    Если пользователя не нашлось, то вернётся его логина не будет среди ключей хеша.
    Если при работе функции произошла ошибка, то ругнется в STDERR и вернёт undef.

=head2 SYNOPSIS

    Пример использования:
    my $mass_staff = get_mass_staff_info(['meridian', 'oxid'], ['login', 'work_phone']);
    my ($login1, $phone1) = ($mass_staff->{'meridian'}->{'login'}, $mass_staff->{'meridian'}->{'work_phone'});
    my ($login2, $phone2) = ($mass_staff->{'oxid'}->{'login'},     $mass_staff->{'oxid'}->{'work_phone'});

=cut

sub get_mass_staff_info {
    my $domain_logins = shift;
    return {} unless $domain_logins;
    $domain_logins = [ $domain_logins ] unless ref($domain_logins);
    
    my $fields = shift || [qw(id uid guid login work_email work_phone)]; # остальные поля не просто строчки, а сложнее: https://staff-api.yandex-team.ru/_static/api/schemas/person.jso
    my $result_data = {};

    unless ( $STAFF_OAUTH_TOKEN ) {
        confess "Not set global variable \$STAFF_OAUTH_TOKEN!\n";
    }
    
    if ( $STAFF_REQUEST_METHOD ne 'GET' ) {
        confess "Request method '$STAFF_REQUEST_METHOD' not supported for Staff API v.3 https://wiki.yandex-team.ru/staff/apiv3 !\n";
    }

    # Так как API поддерживает на данный момент только GET метод, следовательно мы ограничены в длине запроса.
    # Расчитаем длину $fields + ESCAPED_COMMA_LENGTH х (на число $fields - 1) для запятых.
    my $fields_length = (scalar(@$fields) - 1)*ESCAPED_COMMA_LENGTH;
    $fields_length   += bytes::length( uri_escape($_) ) for (@$fields);

    # Вычислим максимальную длину логина в $domain_logins + ESCAPED_COMMA_LENGTH на запятую.
    my $max_login  = max_by { bytes::length( uri_escape($_) ) } @$domain_logins;
    my $max_length = bytes::length( uri_escape($max_login) ) + ESCAPED_COMMA_LENGTH;
    
    # А теперь расчитаем по сколько логинов максимум за раз мы можем посылать в запросе, что бы не превысить
    # ограничение на максимум $STAFF_MAX_SIZE_URL байт в запросе. Для этого вычтем из $STAFF_MAX_SIZE_URL длину
    # обязательной части запроса в виде длины адреса сервера, имён параметров и списка запрашиваемых полей и
    # разделим остаток на длину максимального по размеру логина. Пачки у нас будут для простоты одинаковые.
    my $persons_url = ($STAFF_OAUTH_URL =~ /\/$/) ? $STAFF_OAUTH_URL.'persons' : $STAFF_OAUTH_URL.'/persons';
    
    my $chunk_size = int( ($STAFF_MAX_SIZE_URL 
                            - bytes::length($persons_url) 
                            - bytes::length('?_page=1&_limit='.LIMIT_PERSON_PER_PAGE.'&_fields='.'&login=') 
                            - $fields_length
                          )/$max_length );

    # warn "length = ".( bytes::length($persons_url.'?_page=1&_limit='.LIMIT_PERSON_PER_PAGE.'&_fields='.'&login=') 
    #   + $fields_length )." \$chunk_size = $chunk_size, \$max_length = $max_length, \$fields_length = $fields_length\n";

    # Этим числом и ограничим число передаваемых логинов за раз. А так как результат API к персоналиям бьётся
    # по LIMIT_PERSON_PER_PAGE объектов на страницу и что бы не разбираться с этими страницами, то ограничим 
    # число логинов второй границей в LIMIT_PERSON_PER_PAGE штук. 
    $chunk_size = LIMIT_PERSON_PER_PAGE if ( $chunk_size > LIMIT_PERSON_PER_PAGE );
    
    for my $chunk ( chunks($domain_logins, $chunk_size) ) {
        my $url = Yandex::HTTP::make_url($persons_url, {
            '_page'     => 1,   # номер страницы с объектами, 1 - начальная
            '_limit'    => LIMIT_PERSON_PER_PAGE, # объектов на страницу
            '_fields'   => join(',', @$fields),
            'login'     => join(',', @$chunk),
        } );
        
        # warn "length \$url = ".bytes::length($url)." \$url = '$url'\n";
        my $content = eval {
            Yandex::HTTP::http_fetch($STAFF_REQUEST_METHOD, $url, {}, 
                headers => { 'Authorization' => 'OAuth '.$STAFF_OAUTH_TOKEN, },
                timeout => 5,
            );
        };

        if ( !defined $content ) {
            cluck "Staff API Error: $@";
            return;
        }

        my $mass_staff = eval { JSON->new->allow_nonref->utf8->decode($content) };
            
        if( $@ 
            || !defined($mass_staff)
            || (ref($mass_staff) ne 'HASH') 
            || !exists($mass_staff->{"result"})
            || (ref($mass_staff->{"result"}) ne 'ARRAY')
        ) {
            cluck "Staff parse error. Could not parse Staff response = ".Dumper($content).
                    " for $STAFF_REQUEST_METHOD request with logins = ('".join("', '", @$chunk)."'),".
                    "\nresult of JSON decode = ".Dumper($mass_staff).
                    "length \$url = ".bytes::length($url)." \$url = '$url'\n".
                    ($@ ? "\nError text of decode: ".$@ : '');
            return;
        }
        else {
            for my $staff ( @{$mass_staff->{"result"}} ) {
                if ( exists($staff->{'login'}) ) {
                    $result_data->{$staff->{'login'}} = $staff;
                }
            }
        }
    }

    return $result_data;
}

=head2 get_staff_info_ext($method; %params)

    Небольшая функция обертка без всяких сложностей и ограничений для получения данных из api staff'а.
    Принимает метод, к которому надо обратиться в api (persons, groups, organizations и т.д.) и
    хэш параметров для данного метода (посмотреть список можно здесь https://api.staff.yandex-team.ru/v3/).
    Возвращает ссылку на массив с полученными данными.
    Если при работе функции произошла ошибка, то ругнется в STDERR и вернёт undef.

=head2 SYNOPSIS

    Пример использования:
    my $data = get_staff_info_ext('persons', (login => 'palasonic', _fields => 'uid,name.first'));
    my ($uid, $first_name) = (@$data[0]->{uid}, @$data[0]->{name}->{first});
    my $groups = get_staff_info_ext('groups', (_fields => 'name,url,id', _limit => 5));

=cut

sub get_staff_info_ext {
    my ($method, %params) = @_;
    return () unless (%params || $method);
    
    my @result_data = ();

    unless ( $STAFF_OAUTH_TOKEN ) {
        confess "Not set global variable \$STAFF_OAUTH_TOKEN!\n";
    }
    
    if ( $STAFF_REQUEST_METHOD ne 'GET' ) {
        confess "Request method '$STAFF_REQUEST_METHOD' not supported for Staff API v.3 https://wiki.yandex-team.ru/staff/apiv3 !\n";
    }
    
    $params{_page} = 1;
    $params{_limit} = LIMIT_PERSON_PER_PAGE unless exists $params{_limit};
    
    my $url = Yandex::HTTP::make_url("$STAFF_OAUTH_URL$method", \%params );
    
    my $content = eval {
        Yandex::HTTP::http_fetch($STAFF_REQUEST_METHOD, $url, {}, 
            headers => { 'Authorization' => 'OAuth '.$STAFF_OAUTH_TOKEN, },
            timeout => 5,
        );
    };

    if ( !defined $content ) {
        cluck "Staff API Error: $@";
        return;
    }

    my $mass_staff = eval { JSON->new->allow_nonref->utf8->decode($content) };
           
    if( $@ 
        || !defined($mass_staff)
        || (ref($mass_staff) ne 'HASH') 
        || !exists($mass_staff->{"result"})
        || (ref($mass_staff->{"result"}) ne 'ARRAY')
    ) {
        cluck "Staff parser error or bad request. ".Dumper($content).
                " for $STAFF_REQUEST_METHOD request ".
                " \$url = '$url'\n".
                ($@ ? "$@" : '');
        return;
    } else {
        for my $staff ( @{$mass_staff->{"result"}} ) {
            push @result_data, $staff;
        }
    }

    return \@result_data;
}

1;

__END__
