package Yandex::StopWords;

=head1 NAME

    Yandex::StopWords

=head1 DESCRIPTION

    Определение минус-слов по словарю
    В файле должно быть минимальное количество зависимостей, чтобы можно было 
    работать на кластере mapreduce или использовать в БК

=cut

use strict;
use warnings;
use feature 'state';
use utf8;

use Cwd qw//;
use Carp qw/croak/;

use parent 'Exporter';
our @EXPORT_OK = qw/is_stopword process_quoted_phrases $STOP_WORDS_FILE $STOP_WORDS_AUTOREFRESH_FILE/;


=head2 $STOP_WORDS_FILE

    путь к файлу со стоп-словами, по-умолчанию - рядом с пакетом

=cut
our $STOP_WORDS_FILE //= Cwd::abs_path(__FILE__ =~ s/StopWords\.pm/stopword.lst/r);

=head2 $STOP_WORDS_AUTOREFRESH_FILE

    путь к автообновляемому файлу со стоп-словами; приоритетный, но может отсутствовать

=cut
our $STOP_WORDS_AUTOREFRESH_FILE //= q{/opt/ppc-data/import/stopwords_autorefresh.lst};

=head2 $STOP_WORDS_CHECK_FILE_MODIFIED_SECONDS
    
    с какой периодичностью проверять, не изменился ли файл

=cut
our $STOP_WORDS_CHECK_FILE_MODIFIED_SECONDS //= 15*60;


# ленивая, кешированная щагрузка словаря
sub _stop_words_dict
{
    state $stop_words_dicts = {};
    state $current_file;
    if (!$current_file || $current_file->{stat_time} + $STOP_WORDS_CHECK_FILE_MODIFIED_SECONDS < time()) {
        $current_file = current_words_file();
    }

    if (!exists $stop_words_dicts->{$current_file->{path}} ||
        $stop_words_dicts->{$current_file->{path}}->{file_mod_time} != $current_file->{mod_time} ) {
        open(my $fh, "<:utf8", $current_file->{path}) || croak "Can't open $current_file->{path}: $!";
        my %dict;
        while(my $line = <$fh>) {
            $dict{_normalize_word($_)} = undef for grep {$_ ne ''} split /\s+/, $line; 
        }
        close($fh) || croak "Can't close $current_file->{path}: $!";
        $stop_words_dicts->{$current_file->{path}} = {file_mod_time => $current_file->{mod_time},
                                                      dict => \%dict};
    }
    return $stop_words_dicts->{$current_file->{path}}->{dict};
}

=head2 current_words_file

    Возвращает более свежий по modtime файл, среди файла приехавшего с пакетов, и автогенерируемого файла.
    Считаем что этот файл содержит актуальные стоп-слова.
    Если ни одного файла не существует - падаем.
    Формат ответа:
    { path    => <file_path>,
      mod_time => <file_modification_timestamp> }

=cut
sub current_words_file {
    my $current_file;

    for my $fpath (grep { $_ && -r $_ } ($STOP_WORDS_FILE, $STOP_WORDS_AUTOREFRESH_FILE)) {
        my $modtime = (stat($fpath))[9];
        if (!$current_file || $current_file->{mod_time} < $modtime) {
            $current_file = {path    => $fpath,
                             mod_time => $modtime,
                             stat_time => time(),};
        }
    }

    croak "Readable files with stopwords is not founded" unless $current_file;
    return $current_file;
}

=head2 is_stopword($word)

    проверить, является ли слово стоп-словом

=cut
sub is_stopword($)
{
    my ($word) = @_;

    my $dict = _stop_words_dict();

    return exists $dict->{_normalize_word($word)};
}

=head2 _normalize_word

    Приводит переданное слово к нормализованному виду, чтобы совпадало со словарём.

=cut

sub _normalize_word {
    my ($word) = @_;

    $word = lc($word);
    $word =~ s/ё/е/g;

    return $word;
}

=head2 process_quoted_phrases($)

    преобразование фразы как при отправке в Баннерную крутилку
    (для фраз в кавычках проставляем + к стоп-словам и дописываем ~0)

=cut
sub process_quoted_phrases($)
{
    my ($phrase) = @_;

    $phrase = '' unless defined $phrase;

    $phrase =~ s/^\s+//;
    $phrase =~ s/\s+$//;

    # если нет кавычек - просто возвращаем фразу
    return $phrase if $phrase !~ /^"[^"]+"$/;

    $phrase =~ s/"//g;

    my @res;
    for my $word (split /\s+/, $phrase) {
        $word = "+$word" if is_stopword(_normalize_word($word)) && $word !~ /^[!+]/;
        push @res, $word;
    }
    return join(' ', @res).' ~0';
}


1;
