#!/usr/bin/perl

=encoding utf8

=head1 NAME

    refreshStopWords.pl

=head1 SYNOPSIS

    refreshStopWords.pl [--beta-refresh-package] [--interactive] [--bs-host bssoap.yandex.ru:81] [--log-path /var/www/yandex] [--max-decrease-pct 20]

=head1 DESCRIPTION

    Скрипт втягивает стоп-слова из БК, сравнивает с теми что лежат локально, если есть изменения - записывает новые слова в локальный файл
    Исходный файл, изменения с которым мы вычисляем - более "свежий"" файл из пакетного списка слов, и файла с автообновленным списком слов

    Опции командной строки:
        --help
            выводит справку по использованию скрипта

        --interactive
            выводит результаты работы скрипта в stdout (прошло ли все успешно, были ли изменения и т.п.)
            полезно когда мы хотим закоммитить изменения, и обновить пакет

        --bs-host <host_domain>
            хост БК, с которого следует тянуть стоп-слова, по-умолчанию bssoap.yandex.ru

        --log-path <path>
            в какую директорию писать логи работы скрипта, по-умолчанию через syslog, с префиксом stopwords (на серверах Директа - /var/log/yandex/stopwords)

        --no-log-required
            не писать логи ни в файл, ни в syslog (нужно для случаев когда в системе не установлен Yandex::Log)
            если установлена переменная окружения LOG_TEE=1 - логи пишутся в STDERR

        --min-stop-words-num <XX>
            предохранитель для борьбы с сомнительными ответами от БК
            указывает какое минимальное количество стоп-слов должно быть в списке после обновления, если показатель не достигнут - пишет в лог ошибку и прерывает выполнение
            по-умолчанию: 200

        --default-file-path <file_path>
            путь к файлу в котором лежит список слов, которые идут с пакетом (по-умолчанию - stopwords.lst рядом с Yandex/StopWords.pm)

        --auto-file-path <file_path>
            путь к файлу в который будут записываться автообновляемые стоп-слова (по-умолчанию - /opt/ppc-data/import/stopwords_autorefresh.lst)

        --create-dir
            создавать ли директорию в которой будет создаваться файл со стоп-словами
            по-умолчанию: если задан --file-path - false, иначе - true

        --beta-refresh-package 
            метаопция, предназначена для запуска с текущей директорией yandex-lib/stopwords/lib/Yandex
            равноценна комбинации таких опций: 
            --interactive 
            --bs-host bssoap.yandex.ru:81
            --default-file-path stopwords.lst
            --auto-file-path stopwords.lst

=cut

use strict;
use warnings;
use utf8;

use open ':std' => ':utf8';

use v5.14;

use Carp;
use JSON;
use Getopt::Long;
use Path::Tiny;
use List::MoreUtils qw/uniq/;

use Yandex::HTTP qw/http_fetch/;
use Yandex::StopWords qw/$STOP_WORDS_FILE $STOP_WORDS_AUTOREFRESH_FILE/;
use Yandex::ListUtils qw/xminus/;

my $INTERACTIVE = 0;
my $BS_HOST = 'bssoap.yandex.ru';
my $BETA_REFRESH_PACKAGE = 0;
my $LOG_PATH;
my $NO_LOG_REQUIRED;
my $MIN_STOP_WORDS_NUM = 200;
my $REL_URL = '/export/export-table.cgi?table_name=Uniword';

my $CUSTOM_STOP_WORDS_AUTOREFRESH_FILE;
my $CUSTOM_STOP_WORDS_FILE;
my $CREATE_DIR;

Getopt::Long::GetOptions(
    'help' => \&usage,
    'interactive' => \$INTERACTIVE,
    'bs-host=s' => \$BS_HOST,
    'beta-refresh-package' => \$BETA_REFRESH_PACKAGE,
    'log-path=s' => \$LOG_PATH,
    'no-log-required' => \$NO_LOG_REQUIRED,
    'min-stop-words-num' => \$MIN_STOP_WORDS_NUM,
    'auto-file-path=s' => \$CUSTOM_STOP_WORDS_AUTOREFRESH_FILE,
    'default-file-path=s' => \$CUSTOM_STOP_WORDS_FILE,
    'create-dir' => \$CREATE_DIR,
)
or die "Getopt error: $!; stop";

if ($BETA_REFRESH_PACKAGE) {
    $INTERACTIVE = 1;
    $BS_HOST = 'bssoap.yandex.ru:81';
    $CUSTOM_STOP_WORDS_AUTOREFRESH_FILE = $CUSTOM_STOP_WORDS_FILE = path($STOP_WORDS_FILE)->basename;
}

# по-умолчанию директорию с файлом автообновления пытаемся создать
# если же задан пользовательский путь к файлу - создаем только по флагу --create-path
if ($CUSTOM_STOP_WORDS_AUTOREFRESH_FILE) {
    $STOP_WORDS_AUTOREFRESH_FILE = path($CUSTOM_STOP_WORDS_AUTOREFRESH_FILE)->absolute;
    $CREATE_DIR //= 0;
} else {
    $CREATE_DIR //= 1;
}

if ($CUSTOM_STOP_WORDS_FILE) {
    $STOP_WORDS_FILE = path($CUSTOM_STOP_WORDS_FILE)->absolute;
}

my $log;

if ($NO_LOG_REQUIRED) {
    $log = new SimpleStdErrLog(no_log => !($ENV{LOG_TEE} || $INTERACTIVE));
} else {
    my $ylog_ok = eval {
        require Yandex::Log;
    };
    unless ($ylog_ok) {
        print STDERR "Error occured when trying to use Yandex::Log. Option --no-log-required might be useful.\n";
        croak $@;
    }

    my %log_opts = (use_syslog => 1, 
                    no_log => 1,
                    syslog_prefix => 'stopwords');
    if ($LOG_PATH) {
        no warnings 'once';
        $Yandex::Log::LOG_ROOT = $LOG_PATH;
        %log_opts = ();
    }

    $log = new Yandex::Log(
        %log_opts,
        log_file_name => "refreshStopWords.log",
        date_suf => "%Y%m%d",
        auto_rotate => 1,
        tee => $ENV{LOG_TEE} || $INTERACTIVE,
    );
}

$log->out("START");

my $new_words = eval { get_new_words() };
if ($@) {
    $log->die("Error while fetching new stopwords from BS: " . $@);
}

my $current_words = eval { get_current_words() };
if ($@) {
    $log->die("Error while reading current stopwords from local file: " . $@);
}

if (scalar(@$new_words) < $MIN_STOP_WORDS_NUM) {
    $log->die(sprintf("Too little count of stopwords recieved: currect_count=%s, new_count=%s, expected_min_count=%s. Suggesting some sort of error on BS side.",
        scalar(@$current_words), scalar(@$new_words), $MIN_STOP_WORDS_NUM));
}

my $added_words = xminus $new_words, $current_words;
my $removed_words = xminus $current_words, $new_words;

if (@$added_words || @$removed_words) {
    $log->out("List of stopwords CHANGED");
    $log->out("stopwords to ADD: " . join(', ', @$added_words)) if @$added_words;
    $log->out("stopwords to REMOVE: " . join(', ', @$removed_words)) if @$removed_words;

    eval { save_words($new_words) };
    $log->die("Error while saving stopwords to local file: " . $@) if $@;
} else {
    eval { update_result_file_mtime() };
    $log->die("Error while updating stopwords local file mtime: " . $@) if $@;

    $log->out("List of stopwords NOT CHANGED");
}

$log->out('FINISH');



    
sub get_new_words {
    my $url = "http://$BS_HOST$REL_URL";
    $log->out("Requesting new stopwords from $url");

    my $new_words_json = http_fetch('GET', $url);
    my @new_words = sort {$a cmp $b} uniq map {$_->{Data}}
                    grep { $_->{Options} && $_->{Options} =~ /(^|,)stop(,|$)/ } 
                    @{ from_json($new_words_json) };
    $log->out(sprintf("Recieved %s stopwords from BS", scalar(@new_words)));
    return \@new_words;
}

sub get_current_words {
    my $current_file = eval { Yandex::StopWords::current_words_file() };
    unless ($current_file) {
        $log->warn("File with current words not founded. Assume that list of current stop words is empty");
        return [];
    }

    $log->out("Reading local stopwords from $current_file->{path}");
    open(my $fh, "<:utf8", $current_file->{path}) || $log->die("Can't open $current_file->{path}: $!");
    my %current_words;
    while(my $line = <$fh>) {
        $current_words{$_} = undef for grep {$_ ne ''} split /\s+/, $line; 
    }
    close($fh) || $log->die("Can't close $current_file->{path}: $!");

    $log->out(sprintf("Read %s stopwords from local file", scalar(keys %current_words)));
    return [sort keys %current_words];
}

sub save_words {
    my ($words) = @_;

    if ($CREATE_DIR) {
        my $dir = path($STOP_WORDS_AUTOREFRESH_FILE)->parent;
        unless (-d $dir) {
            $log->out("Creating directory '$dir' for autorefreshed file");
            eval { path($dir)->mkpath };
            if ($@) {
                $log->die("Can't create directory '$dir' for autorefreshed file: $@");
            }
        }
    }
    my $tmp_file = $STOP_WORDS_AUTOREFRESH_FILE . '.tmp';


    $log->out("Saving stopwords to $tmp_file");

    open(my $fh, ">:utf8", $tmp_file) || $log->die("Can't open $tmp_file for writing: $!");
    print $fh join("\n", @$words);
    close($fh) || $log->die("Can't close $tmp_file: $!");

    rename($tmp_file, $STOP_WORDS_AUTOREFRESH_FILE) || $log->die("Can't move $tmp_file to $STOP_WORDS_AUTOREFRESH_FILE : $!");
    $log->out("Stopwords successfully saved to $STOP_WORDS_AUTOREFRESH_FILE");
}

sub update_result_file_mtime {
    $log->out("Updating mtime of $STOP_WORDS_AUTOREFRESH_FILE");
    utime(undef, undef, $STOP_WORDS_AUTOREFRESH_FILE);
}

sub usage {
    my $base_cmd = "podselect -section NAME -section SYNOPSIS -section USAGE -section DESCRIPTION -section RUNNING $0 | /usr/local/bin/pod2text-utf8";

    if ( my $pager = $ENV{PAGER} ) {
        system("$base_cmd | $pager");
    } else {
        system($base_cmd);
    }

    exit(0);
}


package SimpleStdErrLog;

##
# класс необходим для отрыва жесткой зависимости пакета от наличия в системе Yandex::Log (с этим есть проблемы на precise)
##

use Carp;
use JSON;
use POSIX qw/strftime/;

sub new {
    my $this = shift;
    my $class = ref($this) || $this;

    my $self = {@_};

    bless $self, $class;

    return $self;
}

sub _get_text_message
{
    my $message = shift;
    if (ref $message) {
        return to_json($message, {
            allow_unknown => 1,
            allow_blessed => 1,
            convert_blessed => 1
        });
    } else {
        return defined $message ? $message : '';
    }
}

sub out {
    my $self = shift;
    return if $self->{no_log};

    my $time = time();
    my $date_str = strftime("%Y-%m-%d\t%H:%M:%S", localtime(time()))."\t";
    for (@_) {
        print STDERR $date_str . _get_text_message($_) . "\n";
    }
}

sub warn {
    my $self = shift;
    
    $self->out(@_);
    carp _get_text_message(@_);
}

sub die {
    my $self = shift;
    
    $self->out(@_);
    croak _get_text_message(@_);
}

1;
