package Yandex::Svn;

# $Id$

=head1 NAME

Yandex::Svn - выполнение некоторых операций с svn

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use base qw/Exporter/;
our @EXPORT = qw/
    svn_files 
    svn_unversioned_files 
    svn_file_status
    svn_properties 
    svn_info
    svn_branches_stat
    /;

use Cwd;
use Encode;
use XML::LibXML;

use Yandex::Shell;

=head2 svn_files(dir, %O)

    Получение списка файлов и директорий, контролируемых svn для одной или нескольких директорий
    Работает рекурсивно или нет в зависимости от параметра depth

    Пропускает externals
    Аргументы позиционные:
      - файл/директория или ссылка на массив файлов/директорий

    Параметры именованные
      depth 
        будет как есть передан svn'у. Нужен, чтобы контролировать рекурсивное/нерекурсивное получение списка
      externals 
        обрабатывать ли externals. По-умолчанию - не обрабатывать

=cut

sub svn_files {
    my ($files, %O) = @_;

    my $status = svn_file_status( $files, externals => $O{externals}, depth => $O{depth} );

    my @bucket_keys = qw( added conflicted merged modified normal replaced );
    if ( $O{externals} ) {
        push @bucket_keys, 'external';
    }

    my @result;
    foreach my $key (@bucket_keys) {
        my $bucket = $status->{$key} || [];
        push @result, @$bucket;
    }

    @result = sort @result;
    return @result;
}

=head2 svn_unversioned_files(dir, %O)

    Рекурсивное получение списка файлов и директорий, неконтролируемых svn для одной или нескольких директорий
    Пропускает externals
    Аргументы позиционные:
      - файл/директория или ссылка на массив файлов/директорий

    Аргументы именованные:
      include_ignored
        включить в список файлы, внесённые в списки в свойствах svn:ignore
      externals
        Обрабатывать ли externals. По умолчанию они не обрабатываются (--ignore-externals).

=cut

sub svn_unversioned_files {
    my ( $files, %O ) = @_;

    my $status = svn_file_status( $files, externals => $O{externals} );

    $status->{unversioned} ||= [];
    $status->{ignored}     ||= [];

    if ( $O{include_ignored} ) {
        return ( @{ $status->{unversioned} }, @{ $status->{ignored} } );
    }

    return @{ $status->{unversioned} };
}

=head2 svn_file_status(dir, %O)

    Получение списка файлов в рабочей копии SVN (API для svn status). Возвращает список файлов,
    сгруппированный по статусу:

        {
            external    => [ filename, filename, ... ],
            ignored     => [ filename, filename, ... ],
            modified    => [ filename, filename, ... ],
            normal      => [ filename, filename, ... ],
            unversioned => [ filename, filename, ... ],
            ...
        }

    Все возможные ключи:
        added conflicted deleted external ignored
        incomplete merged missing modified none
        normal obstructed replaced unversioned
    (http://svn.apache.org/viewvc/subversion/trunk/subversion/svn/schema/status.rnc?view=markup)

    Позиционные параметры:
      - Файл/каталог или список файлов/каталогов.

    Именованные параметры:
      depth
        Будет как есть передан svn. Нужен, чтобы контролировать рекурсивное/нерекурсивное получение списка.
      externals
        Обрабатывать ли externals. По умолчанию они не обрабатываются (--ignore-externals).
      no_verbose
        Не передавать опцию -v (по умолчанию передаётся).

=cut

sub svn_file_status {
    my ( $files, %O ) = @_;

    unless ( ref $files eq 'ARRAY' ) {
        $files = [$files];
    }

    my @svn_opts = (
        '--no-ignore',
        '--xml',
    );

    unless ( $O{no_verbose} ) {
        push @svn_opts, '-v';
    }

    unless ( $O{externals} ) {
        push @svn_opts, '--ignore-externals';
    }

    if ( my $depth = $O{depth} ) {
        push @svn_opts, '--depth' => $depth;
    }

    my $xml = yash_qx( 'svn', 'st', @svn_opts, @$files );
    my $doc = XML::LibXML->new()->parse_string($xml)->documentElement();

    my @entries = $doc->findnodes('/status/target/entry');

    my %result;
    foreach my $entry (@entries) {
        my $path   = $entry->getAttribute('path');
        my $status = $entry->findvalue('wc-status/@item');

        my $bucket = ( $result{$status} ||= [] );
        push @$bucket, $path;
    }

    return \%result;
}

=head2 svn_properties

    Рекурсивное получение svn-свойств всех файлов
    Аргументы:
      - файл/директория или ссылка на массив файлов/директорий
    Результат:
      - ссылка на хэш - по полному имени файла - хэш со свойствами

=cut

sub svn_properties {
    my ($files) = @_;
    my $xml = yash_qx("svn", "proplist", "-vR", "--xml", (ref $files eq 'ARRAY' ? @$files : $files));
    my $xml_doc = XML::LibXML->new()->parse_string($xml)->documentElement();            

    my %ret;
    for my $e ($xml_doc->findnodes("/properties/target")) {
        my $path = $e->getAttribute('path');
        $ret{$path} = {map {$_->getAttribute('name')=>$_->textContent} $e->findnodes('./property')};
    }

    return \%ret;
}

=head2 svn_info

    Базовая информация о рабочей копии
    Аргументы:
      - файл/директория
    Результат:
      - ссылка на хэш со свойствами
        url, revision, repository_root, 
        wc_root -- абсолютный путь к корню рабочей копии (там, где лежит .svn)

=cut

sub svn_info {
    my ($file) = @_;
    my $xml = yash_qx("svn", "info", "--xml", $file);
    my $xml_doc = XML::LibXML->new()->parse_string($xml)->documentElement();            

    my %ret = (
        url             => $xml_doc->findvalue('/info/entry/url/text()'),
        revision        => $xml_doc->findvalue('/info/entry/@revision'),
        last_change_rev => $xml_doc->findvalue('/info/entry/commit/@revision'),
        repository_root => $xml_doc->findvalue('/info/entry/repository/root/text()'),
        wc_root         => $xml_doc->findvalue('/info/entry/wc-info/wcroot-abspath/text()'),
        );
    return \%ret;
}


=head2 svn_branches_stat

    Статистика по бранчам в репозитории

    Предполагается, что у репозитория стандартное расположение каталогов: 
        /trunk
        /branches
            /branch_1
            /branch_2
    Но если в $svn_root передан url Аркадии ('svn+ssh://arcadia.yandex.ru/arc'), то используется расположение
        /trunk/arcadia/perl/direct
        /branches/direct/perl
            /branch_1
            /branch_2

    Параметры позиционные: 
        $svn_root -- корень репозитория ('svn+ssh://svn.yandex.ru/direct')

    Параметры именованные 
        branches -- список бранчей, по которым нужна статистика
            если параметр отсутствует, или имеет ложное значение --
            возвращаются данные по всем бранчам из branches
            Бранчи могут быть указаны и полным url'ом, и относительно branches

        process_branch_stat -- ссылка на функцию, 
            вызывается, когда получены данные по очередному бранчу.
            В качестве аргумента функции передается хеш с данными по бранчу.
            Если параметр отсутствует или имеет ложное значение -- ничего не делается

    Возвращаемое значение
    ссылка на массив хешей: 
    [
    {
        'branch'        => 'add_new_mypage', # имя бранча
        'create_author' => 'mirage',         # кто создал
        'create_date'   => '2011-05-23',     # когда создан
        'create_rev'    => '22984',          # в какой ревизии создан
        'edit_author'   => 'metallic',       # автор последней правки
        'edit_date'     => '2011-06-15',     # дата последней правки
        'edit_rev'      => '23584',          # ревизия последней правки
        'merged'        => 1,                # был ли смержен в транк
        'merged_rev'    => '23584',          # макс. смерженная в транк ревизия
     },
    ...
    ]

    Примеры 
    my $stat = svn_branches_stat(
        'svn+ssh://svn.yandex.ru/direct', 
        branches => (@ARGV ? \@ARGV : ''),
        process_branch_stat => ($delayed_print ? sub {print ".";} : \&print_branch_stat)
    );

=cut

sub svn_branches_stat
{
    my ($svn_root, %O) = @_;
    my $is_arcadia = $svn_root eq 'svn+ssh://arcadia.yandex.ru/arc';
    my $relative_trunk_url = $is_arcadia ? '/trunk/arcadia/direct/perl' : '/trunk';
    my $relative_branches_url = $is_arcadia ? '/branches/direct/perl' : '/branches';
    my $trunk_url = $svn_root . $relative_trunk_url; 
    my $branches_url = $svn_root . $relative_branches_url;

    # какие бранчи надо обработать: если дан список -- их, иначе -- все
    my @branches; 
    if ($O{branches}){
        @branches = @{$O{branches}};
        s!^\Q$branches_url\E/!! for @branches;
    } else {
        @branches = split "\n", Encode::decode_utf8(yash_qx("svn", "ls", $branches_url));
    }
    s!/$!! for @branches;

    # что было смержено в транк:
    # $MERGED{$branch} = <макс. ревизия из бранча, смерженная в транк>
    my %MERGED;
    for my $s (split ' ', yash_qx("svn", "propget", "svn:mergeinfo", $trunk_url)) {
        $s =~ s!^\Q$relative_branches_url\E!!;
        $s =~ s!^/!!;
        $s =~ /(.*):(?:.*[^\d])?(\d*)$/;
        print STDERR "strange mergeinfo: $s\n" if !$s || !$1 || !$2;
        $MERGED{$1} = $2;
    }

    my @stat;
    for my $br ( @branches ){
        my $st;
        eval {
            $st = yash_qx("svn log -q --stop-on-copy $branches_url/$br");
        };
        if ($@) {
            print STDERR "could not stat $br\n";
            next;
        }
        my @svnlog = grep { !/^-/ } split "\n", $st;

        # первая и последняя записи из лога
        my @create = split /[|\s]+/, $svnlog[-1];
        my @edit   = split /[|\s]+/, $svnlog[0];

        my $s = {
            branch => $br, 
            create_rev => $create[0],
            create_date => $create[2],
            create_author => $create[1],
            edit_rev => $edit[0],
            edit_date => $edit[2],
            edit_author => $edit[1],
            merged => ($MERGED{$br}?1:0),
            merged_rev => $MERGED{$br} || 0,
        };
        $s->{$_} =~ s/^r// for qw/create_rev edit_rev merged_rev/;

        # особый случай: бранч был смержен, удален, создан новый с тем же именем
        # -- не считаем смерженным
        if ($s->{merged} && $s->{create_rev} > $s->{merged_rev}){
            $s->{merged} = 0;
            $s->{merged_rev} = 0;
        }

        push @stat, $s;

        if($O{process_branch_stat}) {
            $O{process_branch_stat}->($s);     
        }
    }

    return \@stat;
}


=head2 verify_working_copy

    Yandex::Svn::verify_working_copy($path);
    Yandex::Svn::verify_working_copy("/var/www/beta.lena-san.8080");

    Проверяет, что $path -- чистая рабочая копия: 
    без локальных модификаций, 
    без неверсионированных файлов (в т.ч. игнорируемых svnst),
    все файлы -- одной ревизии
    все externals заданы с ревизией, и файлы соответствуют этим ревизиям

    Если что-то не так -- умирает с человекопонятным сообщением

=cut

sub verify_working_copy
{
    my ($path, %O) = @_;
    my $cwd = getcwd;
    chdir $path or die "can't cd $path";

    my $svn_info = svn_info(".");
    my $externals_revisions = _externals_revisions(".");

    my @status = split /\r?\n/, yash_qx("svn", "st", "--no-ignore", "-v", ".");
    my @modified;
    my @bad_revisions;
    my $revision = $svn_info->{revision};
    for my $line (@status){
        next unless $line;
        if( $line =~ /^Performing status on external item at '(.*?)'/ ) {
            my $p = "$svn_info->{wc_root}/$1";
            $revision = $externals_revisions->{$p} or die "can't find revision for external item $p";
            next;
        }
        if ( $line =~ /^([^ ])\s+(\S+)/ ){
            my ($st, $file) = ($1, $2);
            if ( $st eq 'X' ){
                next
            } elsif ( $st eq "I" && grep { $file eq $_ } @{$O{skip} || []} ) {
                next
            } else {
                push @modified, $line;
                next;
            }
        }
        $line =~ s/^\s+//;
        my ($rev, undef, undef, $file)= split /\s+/, $line;
        die $line unless $rev;
        if ( $rev != $revision ){
            push @bad_revisions, "r$rev $file";
        }
    }
    if ( @modified > 0 ){
        die "Working copy in $path has modified or unversioned files:\n".join("\n", @modified)."\n";
    }
    if ( @bad_revisions > 1 ){
        die "Working copy in $path (r$svn_info->{revision}) contains files of different revision, at least:\n".join("\n", @bad_revisions)."\n";
    }

    chdir $cwd or die "can't cd $cwd";
}


=head2 _externals_revisions

    Возвращает ссылку на хеш с ревизиями externals-каталогов: 

    {
        path1 => $rev1, 
        path2 => $rev2,
    }

    Если какой-то externals объявлен без ревизии -- умирает.

=cut

sub _externals_revisions
{
    my ($path) = @_;

    my $xml = yash_qx("svn", "propget", "-R", "--xml", "svn:externals", $path);
    my $doc = XML::LibXML->new()->parse_string($xml)->documentElement();
    my @targets = $doc->findnodes("/properties/target");

    my $old_externals_format = qr!^([\w\.\-/]+)\s+-r(\d+)\s+.*!;
    my $new_externals_format = qr!^\s*-r\s*(\d+)\s+[\w\.\-\/\:]+\s+([\w\.\-/]+)\s*!;

    my %rev;
    for my $t (@targets) {
        my $path = $t->getAttribute('path');
        my $value = $t->findvalue("./property/text()");
    
        for my $line ( split /\n+/, $value ){
            if ( $line =~ $old_externals_format ){
                $rev{"$path/$1"} = $2;
            } elsif ( $line =~ $new_externals_format ) {
                $rev{"$path/$2"} = $1;
            } else {
                die "unknown externals format on path $path:\n$value.";
            }

        }
    }
    return \%rev;
}

1;
