package Yandex::SafeSWF;

=pod
    $Id$
    Модуль пытается определить, безопасен ли SWF файл
=cut

use strict;
use warnings;
use Path::Tiny;
use Data::Dumper;
use LWP::UserAgent;

use Yandex::Flasm;
use Yandex::Shell;

use base qw/Exporter/;
our @EXPORT = qw/
    safeswf_check_file
    safeswf_check_url
    safeswf_check_data
    /;

# операторы
# значение - влияние на стек
# undef - влияние неизвестно
my %KNOWN_OPERS = (
    branch => undef, branchIfTrue => undef,
    allowDomain => undef,
    scriptLimits => 0,
    getTimer => 1,
    new => 1, dup => 1, delete => undef, delete2 => undef, swap => 0,
    typeof => 0, toNumber => 0, toString => 0, int => 0,
    initArray => sub {return $_[0]->[-1] =~ /^!(\d+)$/ ? (-$1-1, 1) : (undef);},
    initObject => undef,
    enumerateValue => undef, enumerate => undef,
    add => [-2, 1], subtract => [-2, 1], divide => [-2, 1], multiply => [-2, 1], modulo => [-2, 1], random => [-1, 1],
    increment => 0, decrement => 0,
    substring => 0, concat => -1, stringLength => 0,
    oldAdd => [-2, 1], oldLessThan => -1, oldEquals => [-2, 1],
    not => 0, and => [-2, 1], or => [-2, 1], lessThan => [-2, 1], greaterThan => [-2, 1], varEquals => -1, equals => -1, stringEq => -1, strictEquals => undef,

    bitwiseAnd => [-2, 1], bitwiseOr => [-2, 1], 
    shiftRight => [-2, 1], shiftLeft => [-2, 1],

    instanceOf => [-2, 1], cast => [-2, 1], throw => -1,

    setProperty => -2, getProperty => -1, getMember => 0, setMember => -2, setRegister => 0,
    duplicateClip => undef, removeClip => -1, attachMovie => undef,
    play => undef, gotoAndPlay => undef, stop => undef, gotoAndStop => undef, gotoFrame => undef,
    nextFrame => undef, prevFrame => undef,

    gotoLabel => undef, loop => undef, trace => 0, return => undef, var => undef,

    startDrag => 0, stopDrag => 0,

    enableDebugger => 0, enableDebugger2 => 0,
    constants => 0, metadata => 0, fileAttributes => 0,
    protect => 0,
    as => 0,
    );
my %BAD_OPERS = map {$_ => 1} qw/
    getURL loadMovie loadVariablesNum loadVariables
    /;
# сколько возвращаем значений
my %KNOWN_METHODS = (
    abs => 1, sin => 1, cos => 1, atan => 1, atan2 => 1, pow => 1, sqrt => 1, round => 1, floor => 1, ceil => 1, isNaN => 1, parseInt => 1,
    random => 1, getTime => 1,
    push => undef, pop => undef,
    substring => undef, substr => undef, charAt => 1, splice => undef, slice => undef, lastIndexOf => 1, split => undef,
    toString => 1,
    toLowerCase => 1, indexOf => 1,
    instanceOf => undef, newMethod => undef,
    getBytesLoaded => 1, getBytesTotal => 1,
    addListener => undef, registerClass => undef,
    setMask => undef, play => undef, gotoAndPlay => undef, stop => undef, gotoAndStop => undef,
    prevFrame => undef, nextFrame => undef, gotoFrame => undef, drawFrame => undef,
    duplicateMovieClip => undef, attachMovie => 3, createEmptyMovieClip => undef, removeMovieClip => undef, changeMovie => undef,
    clearInterval => undef, setInterval => "-2+1", bubbleAction => undef,
    lineTo => undef,
    ASSetPropFlags => undef, super => undef,
#    allowDomain => undef,
    );
my %UNSAFE_METHODS = map {$_ => 1} qw(
    fscommand
    getURL
    onRelease
    ExternalInterface
    loadMovie loadMovieNum loadVariables loadVariablesNum
    send sendAndLoad
    unloadMovie unloadMovieNum
    load loadClip loadSound loadPolicyFile
    connect
    );
my %UNSAFE_GET_MEMBER = map {$_ => 1} qw(
    broadcastMessage
);

my %PROPS;
my $g_line;
my $links_re;

# а безопасен ли данный конкретный swf?
sub safeswf_check_url {
    my ($url, $links_limit) = @_;

    my $ua = LWP::UserAgent->new(timeout => 10);
    my $resp = $ua->get($url);
    if (!$resp->is_success) {
        die "Can't get $url: ".$resp->status_line;
    }

    return safeswf_check_data($resp->content, $links_limit);
}

# а безопасен ли данный конкретный swf?
sub safeswf_check_file {
    my ($file, $links_limit) = @_;
    return safeswf_check_data(path($file)->slurp, $links_limit);
}

# а безопасен ли данный конкретный swf?
# $additional_params может быть, например, 
# {
#   video => { vars => ['vid1', 'video1']}, # если здесь пустой список или пустой хэш, то не учитываем название переменной при проверке
#   sound => {}
# }
sub safeswf_check_data {
    my $data = shift;
    my $links_limit = shift;
    my $additional_params = shift;
    my @errors;

    # чекаем код
    my $as_code = flasm_decompile_data($data);
    my $code = flasm_parse_data($as_code, $additional_params);
    push @errors, safeswf_check_code($code, $links_limit, $additional_params);

    # чекаем теги
    my %bad_tags;
    # пробуем SafeSWF::Mill
    if (eval {require Yandex::SafeSWF::Mill; Yandex::SafeSWF::Mill::is_swfmill_available()}) {
        %bad_tags = Yandex::SafeSWF::Mill::swfmill_check_data($data);
    } elsif (eval {require Yandex::SafeSWF::Dump; Yandex::SafeSWF::Dump::is_swfdump_available()}) {
        # пробуем SafeSWF::Dump
        %bad_tags = Yandex::SafeSWF::Dump::swfdump_check_data($data);
    } else {
        push @errors, _err('CANT_CHECK_TAGS', val => "Can't check tags - swfmill & swfdump is not available");
    }
        
    push @errors, map {_err('UNKNOWN_TAG', val => "Tag $_ occured $bad_tags{$_} times")} keys %bad_tags;

    return @errors;
}

# инициализация глобальных переменных
sub _init_check {
    my ($links_limit) = shift;
    $links_limit = 1 if ! defined $links_limit;
    die "incorrect links limit" if $links_limit !~ /^\d+$/ || !$links_limit;
    %PROPS = ();
    $g_line = 0;
    $links_re = "(".join('|', 1..$links_limit).")";
}

sub safeswf_check_code {
    my ($code, $links_limit, $additional_params) = @_;
    _init_check($links_limit);
    my @errors = _check_code($code, undef, $additional_params);
    $g_line = {pos => 0};
    push @errors, _err('NO_GOOD_CLICK') if !$PROPS{GoodClick};
    return @errors;
}

# собственно проверка кода. хе-хе-хе.
sub _check_code {
    my ($src_code, $parents, $additional_params) = @_;
    $parents ||= [];
    # массив для ошибок
    my @ret;
    # стек
    my @stack = ();
    my %regs;
    my @labels;
    my @good_clicks;
    for my $line (@$src_code) {
        my ($type, $cmd, $pos, $params, $code) = map {$line->{$_}} qw/type cmd pos params code/;
        $g_line = $line;
        if ($type eq 'pattern') {
            if ($params->{good_click}) {
                $PROPS{GoodClick} = 1;
            }

            if( exists $params->{video} and exists $additional_params->{video} ){
                    push @ret, _err("VIDEO VARNAME IS INCORRECT OR ISN'T USING THIS NetConnection", val => undef) unless $params->{video};
            } elsif( exists $params->{video} ) {
                my $row_number = $g_line->{pos} + $params->{lines} - 3;
                push @ret, _err("UNSAFE_METHOD", val => 'connect', 'row_number' => $row_number );
            }

            if(exists $params->{sound} and not exists $additional_params->{sound}){
                    push @ret, _err("STRANGE_FUNCTION_ARGC", val => 'Sound');
            }

            @stack = ();
            %regs = ();
        } elsif ($type eq 'cont') {
            # контейнер проверяем отдельно, сохраняя его в истории
            my @subparents = (@$parents, {cmd => $cmd, params => $params, stack => [@stack]});
            push @ret, _check_code($code, \@subparents, $additional_params);
            if ($cmd =~ /^( function | function2 )$/x) {
                push @stack, "?";
            } else {
                @stack = ();
                %regs = ();
            }
        } elsif ($type eq 'label' || $type eq 'catch') {
            # при переходе сбрасываем стек
            @stack = ();
            %regs = ();
            push @labels, $params;
        } elsif ($type eq 'unknown') {
            @stack = ();
            %regs = ();
            if ($cmd eq 'tag' && $params =~ /^( 63 | 88 )$/x) {
                # пропускаем знакомые теги
            } else {
                push @ret, _err('UNKNOWN_FLASM', val => "$cmd: $params");
            }
        } elsif ($type eq 'oper') {
            #push @ret, _err("TST_STACK", val => join(', ', @stack));
            if ($cmd eq 'push') {
                for my $v (Yandex::Flasm::_parse_params($params)) {
                    if ($v =~ /^r:\d+$/) {
                        if (exists $regs{$v}) {
                            push @stack, $regs{$v};
                        } else {
                            push @stack, "unknown-$v";
                            #push @ret, _err('NO_SUCH_REGISTER', val => "$v");
                        }
                    } else {
                        push @stack, "!$v";
                    }
                }
                #print STDERR Dumper [\@stack, $params];
            } elsif ($cmd eq 'pop') {
                pop @stack;
            } elsif ($cmd eq 'setRegister') {
                if (!@stack) {
                    $regs{$params} = 'unknown-stack';
                } else {
                    $regs{$params} = $stack[-1];
                }
            } elsif ($cmd eq 'getVariable') {
                if (@stack) {
                    $stack[-1] = "=$stack[-1]";
                }
            } elsif ($cmd eq 'getMember') {
                if (@stack >= 2) {
                    my ($obj, $name) = splice(@stack, -2);
                    push @stack, "~$obj.$name";
                    if ($name !~ s/^!'(\w+)'$/$1/) {
                        push @ret, _err("STRANGE_GET_MEMBER_NAME", val => $name);
                    } else {
                        if (exists $UNSAFE_METHODS{$name} || exists $UNSAFE_GET_MEMBER{$name}) {
                            push @ret, _err("UNSAFE_GET_MEMBER_NAME", val => $name);
                        }
                    }
                } else {
                    push @ret, _err("STRANGE_GET_MEMBER_STACK", val => join(' ', @stack));
                    @stack = ();
                    %regs = ();
                }
            } elsif ($cmd =~ /^( setVariable | setMember | varEquals )$/x) {
                if ($cmd =~ /^(setVariable|varEquals)$/ && @stack >= 2 || $cmd eq 'setMember' && @stack >= 3) {
                    my $val = pop @stack;
                    my $name = pop @stack;
                    if ($cmd eq 'setMember') {
                        my $obj = pop @stack;
                    }
                    if ($name =~ /link$links_re/) {
                        push @ret, _err("STRANGE_LINK$1_MODIFICATION");
                    } elsif ($name =~ /^( on | release | onClipEvent | onClipEvent )$/x) {
                        push @ret, _err("STRANGE_VARMEM_MODIFICATION", val => "$name <- $val");
                    } elsif ($name =~ /htmlText/x) {
                        push @ret, _err("USAGE_OF_HTML", val => "$name <- $val");
                    } elsif ($name !~ /^!'?[\w:\.]+'?$/) {
                        push @ret, _err("STRANGE_SET_VARMEM_NAME", val => "$name <- $val");
                    }
                } else {
                    push @ret, _err("STRANGE_SET_VARMEM_STACK", val => join(' ', @stack));
                    @stack = ();
                    %regs = ();
                }
            } elsif ($cmd =~ /^( newMethod )$/x) {
                if (@stack >= 3) {
                    my $name = pop @stack;
                    my $pack = pop @stack;
                    my $argc = pop @stack;
                    $argc =~ s/^!//;
                    if ($argc !~ /^(0\.0|\d+)$/) {
                        push @ret, _err('STRANGE_NEWMETHOD_ARGC', val => $argc);
                        @stack = ();
                        %regs = ();
                    } elsif (@stack < $argc) {
                        push @ret, _err('STRANGE_NEWMETHOD_ARGC_STACK', val => $argc);
                        @stack = ();
                    } else {
                        _modify_stack(\@stack, -$argc, 1);
                    }
                } else {
                    push @ret, _err('STRANGE_NEWMETHOD_STACK', val => join(' ', @stack));
                    @stack = ();
                    %regs = ();
                }
            } elsif ($cmd =~ /^( initObject )$/x) {
                if (@stack >= 1) {
                    my $argc = pop @stack;
                    $argc =~ s/^!//;
                    if ($argc !~ /^(0\.0|\d+)$/) {
                        push @ret, _err('STRANGE_INITOBJECT_ARGC', val => $argc);
                        @stack = ();
                    } elsif (@stack < $argc*2) {
                        push @ret, _err('STRANGE_INITOBJECT_ARGC_STACK', val => $argc);
                        @stack = ();
                    } else {
                        _modify_stack(\@stack, -2*$argc, 1);
                    }
                } else {
                    push @ret, _err('STRANGE_INITOBJECT_STACK', val => join(' ', @stack));
                    @stack = ();
                    %regs = ();
                }
            } elsif ($cmd eq 'getURL' && $params =~ /^'FSCommand:(\w+)' '(.*)'$/) {
                # отлов FSCommand
                push @ret, check_fscommand($1, $2);
            } elsif ($cmd eq 'getURL2') {
                # переход по урлу
                if (@stack < 2) {
                    push @ret, _err('STRANGE_GETURL_STACK', val => join(' ', @stack));
                }
                my ($href, $target) = @stack[-2, -1];
                if ($href =~ /^!'FSCommand:(\w+)'$/i) {
                    # fscommand
                    my $fscmd = $1;
                    $target =~ s/^!//;
                    push @ret, check_fscommand($fscmd, $target);
                } else {
                    # стандартное описание события
                    push @ret, check_click($parents, \@stack, \@labels, \@good_clicks);
                }
            } elsif ($cmd =~ /^( callFunction | callMethod )/x) {
                # вызов метода
                if ($cmd eq 'callFunction' && @stack >= 2 || $cmd eq 'callMethod' && @stack >= 3) {
                    # получаем название метода
                    my $fname = pop @stack;
                    my $obj = $cmd eq 'callMethod' ? pop(@stack) : '';
                    my $argc = pop @stack;
                    $argc =~ s/^!//;
                    if ($argc !~ /^(0\.0|\d+)$/) {
                        push @ret, _err('STRANGE_FUNCTION_ARGC', val => $argc);
                    } elsif (@stack < $argc) {
                        push @ret, _err('STRANGE_FUNCTION_ARGC_STACK', val => $argc);
                    } else {
                        if ($fname =~ /^!'(\w+)'$/i) {
                            if ($1 eq 'getURL' && $argc == 2) {
                                # меняем порядок в стеке
                                push @ret, check_click($parents, [$stack[-1], $stack[-2]], \@labels, \@good_clicks);
                            } elsif (exists $UNSAFE_METHODS{$1}) {
                                push @ret, _err('UNSAFE_METHOD', val => $1);
                            } elsif ($1 eq 'broadcastMessage' && $stack[-1] eq "!'onRelease'") {
                                push @ret, _err('UNSAFE_BROADCASTMESSAGE', val => $stack[-1]);
                            }
                            _modify_stack(\@stack, -$argc, 1);
                        } else {
                            push @ret, _err('STRANGE_METHOD', val => $fname);
                            @stack = ();
                            %regs = ();
                        }
                    }
                } else {
                    push @ret, _err('STRANGE_FUNCTION_CALL');
                }
            } elsif (exists $BAD_OPERS{$cmd}) {
                push @ret, _err('BAD_OPER', val => $cmd);
            } elsif (exists $KNOWN_OPERS{$cmd}) {
                # если известный оператор - модифицируем стек
                if (ref $KNOWN_OPERS{$cmd} eq 'CODE') {
                    _modify_stack(\@stack, $KNOWN_OPERS{$cmd}->(\@stack) );
                    #print STDERR Dumper \@stack;
                } elsif (ref $KNOWN_OPERS{$cmd} eq 'ARRAY') {
                    _modify_stack(\@stack, @{$KNOWN_OPERS{$cmd}});
                } else {
                    _modify_stack(\@stack, $KNOWN_OPERS{$cmd} );
                }
            } else {
                # лучше перебдеть!
                push @ret, _err('UNKNOWN_OPER', val => $cmd);
            }
        } else {
            push @ret, _err('UNKNOWN_TYPE', val => $type);
        }
    }
    return @ret;
}

sub _err {
    my ($err, %add) = @_;
    return {err => $err, pos => $add{row_number} || $g_line->{pos}, %add};
}

sub _modify_stack {
    my ($stack, @nums) = @_;
    if (!@nums) {
        @$stack = ();
    } else {
        for my $num (@nums) {
            if (!defined $num) {
                @$stack = ();
            } elsif ($num > 0) {
                for (1..$num) {
                    push @$stack, "?";
                }
            } elsif ($num < 0) {
                if (@$stack > abs($num)) {
                    splice @$stack, $num;
                } else {
                    @$stack = ();
                }
            }
        }
    }
}

sub check_fscommand {
    my ($fscmd, $val) = @_;
    if ($fscmd =~ /^(fullscreen|allowscale|showmenu)$/i && lc($val) eq 'false') {
        # ok
        return ();
    } else {
        # переход по урлу
        return _err('FSCOMMAND', val => "$fscmd = $val");
    }
}

sub check_click {
    my ($parents, $stack, $labels, $good_clicks) = @_;
    my $is_good_event =
        (grep { $_->{cmd} =~ /^( on | onClipEvent )$/x
                    && $_->{params} =~ /^( overDownToOverUp | release )$/x
         } @$parents)
        && (grep {$_->{cmd} =~ /^( defineButton | placeMovieClip )$/x} @$parents);
    my $is_good_href = $stack->[-2] =~ /^( =!'link$links_re' | ~=!'_root'.!'link$links_re' | =!'_root\.link$links_re' )$/x;
    my $is_good_target = $stack->[-1] =~ /^!'(_blank|)'$/;
    if (!$is_good_event) {
        return _err('BAD_GETURL_POSITION');
    } elsif (@$labels) {
        return _err('CLICK_AFTER_LABEL');
    } elsif (@$good_clicks) {
        return _err('CLICK_DOUBLE');
    } elsif (!$is_good_target) {
        return _err('INCORRECT_CLICK_TARGET', val => join(' ', $stack->[-1]));
    } elsif (!$is_good_href) {
        return _err('INCORRECT_CLICK_HREF', val => join(' ', $stack->[-2]));
    } else {
        $PROPS{GoodClick} = 1;
        push @{$good_clicks}, $g_line->{pos};
    }
    return ();
}

1;
