package Yandex::Tanker;

# $Id$

=head1 NAME

Yandex::Tanker

=head1 DESCRIPTION

Общение с танкером (работа с форматами и пр.)

=cut


use strict;
use warnings;

use XML::LibXML;

use Yandex::ListUtils;

use utf8;

our $L10N_API_URL       ||= 'http://tanker-api.tools.yandex.net:3000';
our $L10N_INTERFACE_URL ||= 'http://tanker.yandex-team.ru/keys/';
our $HTTP_TIMEOUT  ||= 120;

=head2 xml_to_po($xml_string)

    Преобразуем xml танкера в структуры для отформатированных po-сообщений
    %OPT:
        get_as_is => 1 # отдавать так же как пришло из танкера, не добавлять кавычки, не квотировать
        skip_keyset => 'email' # пропускать кейсет

    Отдаем ссылку на хэш по языкам:
    {
        en => [ {msgid => '...', msgstr => '...'}, ],
        ua => [ {msgid => '...', msgstr => '...'}, ],
    }

=cut

sub xml_to_po
{
    my $xml = shift;
    my %OPT = @_;

    my %messages_by_lang = ();

    my $doc = XML::LibXML->new()->parse_string($xml);
    my @keyset_nodes = $doc->getElementsByTagName('keyset');
    foreach my $keyset_node (@keyset_nodes) {
        my $keyset = $keyset_node->getAttribute('id');
        if ($OPT{skip_keyset} && $OPT{skip_keyset} eq $keyset) {
            next;
        }

        my @key_nodes = $keyset_node->getChildrenByTagName('key');
        foreach my $key_node (@key_nodes) {
            my $msgid = $key_node->getAttribute('id');
            my @translation_nodes = $key_node->getChildrenByTagName('value');
            foreach my $translation_node (@translation_nodes) {
                next if $translation_node->getAttribute('status') ne 'approved';
                my $lang = $translation_node->getAttribute('language');
                $lang = 'ua' if $lang eq 'uk';
                my $msgstr;
                if (my ($plural_node) = $translation_node->getChildrenByTagName('plural')) {
                    # !!! TODO поддержка множественных форм
                    my ($one_node) = $plural_node->getChildrenByTagName('one');
                    $msgstr = $one_node->textContent();
                } else {
                    $msgstr = $translation_node->textContent();
                }
                if (!utf8::is_utf8($msgid)) {
                    utf8::decode($msgid);
                }
                if (!utf8::is_utf8($msgstr)) {
                    utf8::decode($msgstr);
                }

                unless ($OPT{get_as_is}) {
                    $msgstr =~ s/\n/\\n/g;
                    $msgstr =~ s/(?<!\\)"/\\"/g;
                    if ($msgid !~ /\\n$/ && $msgstr =~ /\\n$/) {
                        $msgstr =~ s/\\n$//;
                    }
                } else {
                    $msgstr =~ s/\\n/\n/g;
                }

                $messages_by_lang{$lang} ||= [];
                push @{ $messages_by_lang{$lang} }, {msgid => $msgid, msgstr => $msgstr};
            }
        }
    }

    foreach my $lang (keys %messages_by_lang) {
        $messages_by_lang{$lang} = [xuniq {$_->{msgid}} @{ $messages_by_lang{$lang} }];
    }
    return \%messages_by_lang;
}


=head2 hashref_to_xml($hashref)

    Преобразуем структуру данных в xml для танкера

    Вход:
    {
        keyset => {
            key => {
                ru => '...',
                en => '...',
            },
        },
    }

=cut

sub hashref_to_xml
{
    my $hashref = shift;
    my $project_node = XML::LibXML::Element->new('project');
    $project_node->setAttribute('id', 'direct');
    foreach my $keyset_id (keys %$hashref) {
        my $keyset_node = XML::LibXML::Element->new('keyset');
        $keyset_node->setAttribute('id', $keyset_id);
        foreach my $key_id (keys %{ $hashref->{$keyset_id} }) {
            my $key_node = XML::LibXML::Element->new('key');
            $key_node->setAttribute('id', $key_id);
            $key_node->setAttribute('is_plural', 'False');
            foreach my $language (keys %{ $hashref->{$keyset_id}->{$key_id} }) {
                my $value_node = XML::LibXML::Element->new('value');
                $value_node->setAttribute('language', ($language eq 'ua' ? 'uk' : $language));
                $value_node->setAttribute('status', 'approved');
                $value_node->appendText($hashref->{$keyset_id}->{$key_id}->{$language});
                $key_node->appendChild($value_node);
            }
            $keyset_node->appendChild($key_node);
        }
        $project_node->appendChild($keyset_node);
    }
    my $root_node = XML::LibXML::Element->new('tanker');
    $root_node->appendChild($project_node);
    my $doc = XML::LibXML::Document->new('1.0', 'utf-8');
    $doc->setDocumentElement($root_node);

    return $doc->toString();
}

1;
