package Yandex::TCPMultiplexor;

# $Id: #

=head1 NAME

    Yandex::TCPMultiplexor

=head1 DESCRIPTION

    Мультиплексор для отправки запросов по tcp

=head1 TODO

    Убрать из входных параметров process_queue коллбэк send_request

=cut

use strict;
use warnings;

use version;
use Carp;

use AnyEvent;
use AnyEvent::DNS;
use AnyEvent::Handle;
use Time::HiRes qw//;

use Yandex::Validate qw/is_valid_ip/;


=head2 process_queue
    
    Блокирующий интерфейс для обработки очереди запросов в асинхронном режиме
    Возвращаем код ошибки или завершаемся успешно (return 0), если вся очередь отработана

    process_queue($items, %options);
    $items - ссылка на массив запросов
    %options:
        callbacks => {
            prepare_request  => формируем запрос, передаем текущий $item (можно не указывать, тогда $item и будет запросом)
            send_request     => отправка запроса, передаем запрос, AnyEvent-хэндл и коллбэк, 
                                который сами подготовили, чтобы очередь двигалась 
                                (надо бы это потом спрятать как-нибудь)
            process_response => обработка запроса (вызывается в вышеупомянутом коллбэке), передаем текущий $item и результат
            on_error, on_timeout, on_eof  => что делать при _, передаем текущий $item, 
                                             если коллбэк вернул ошибку (истинное значение), передаем ее наверх и останавливаем выполнение
            on_bad_response  => аналогично для невалидных ответов (при которых упал process_response)
        },
        host => хост (hostname или ip),
        port => порт,
        max_requests => максимальное число одновременных запросов, если не указано, то работаем в последовательном режиме
        timeout => значение таймаута в секундах, дефолт на 3 секунды
        
        retry_interval => таймаут, после которого считаем запрос "тормознутым", и пытаемся перепослать
        num_retries => количество повторных попыток
        resolve_type => как резолвить хостнейм (a, aaaa, any); передается в качестве qtype в AnyEvent::DNS::resolver::resolve. 
            По умолчанию 'aaaa' (резолвить ipv6-адрес).
            Правильнее всего полагаться на умолчальное поведение (ipv6 auf ewig).
        
=cut

sub process_queue 
{
    my ($items, %options) = @_;
    my ($prepare_request, $send_request, $process_response) = map { $options{callbacks}->{$_} } qw/prepare_request send_request process_response/;
    $options{max_requests} ||= 1;
    $options{timeout} ||= 3;

    # если очередь пуста, никто не сделает next_step и send не произойдет
    return 0 unless @$items;

    unless (is_valid_ip($options{host})) {
        my $resolver = AnyEvent::DNS->new(
            untaint => 1,
            exists $ENV{PERL_ANYEVENT_MAX_OUTSTANDING_DNS} ? (max_outstanding => $ENV{PERL_ANYEVENT_MAX_OUTSTANDING_DNS}*1 || 1) : (),
            timeout => [$options{timeout}],
        );
        exists $ENV{PERL_ANYEVENT_RESOLV_CONF} ? length $ENV{PERL_ANYEVENT_RESOLV_CONF} && $resolver->_parse_resolv_conf_file ($ENV{PERL_ANYEVENT_RESOLV_CONF}) : $resolver->os_config;

        my $resolve_cv = AnyEvent->condvar();
        $resolver->resolve($options{host}, $options{resolve_type}||'aaaa', $resolve_cv);
        if (my $resolve_result = $resolve_cv->recv()) {
            my $is_anyevent_new = version->new($AnyEvent::VERSION) >= version->new("6.01");
            $options{host} = $resolve_result->[$is_anyevent_new ? 4 : 3];
        } else {
            die "Could not resolve hostname $options{host}\n";
        }
    }

    my $cv = AnyEvent->condvar();

    my $req_count = @$items;
    my @xqueue = map {{
            position => $_,
            attempts_left => 1 + ($options{num_retries} || 0),
        }}
    ( 0 .. $#$items );

    my $requests_in_progress = 0;
    my $items_ready = 0;

    # переменные приходится объявлять заранее, т.к. их определения зависят друг от друга
    my ($process_queue, $next_step);

    # сюда будем складывать хэндлы, чтобы потом сделать им destroy
    my @handles;

    my @stats;

    # все будет происходить здесь
    $process_queue = sub {
        while ($requests_in_progress < $options{max_requests} && @xqueue) {
            my $item_info = shift @xqueue;

            # если раньше зависало, но уже пришёл результат - пропускаем
            if ( $item_info->{is_ready} || $item_info->{attempts_left} < 1 ) {
                $req_count --;
                next;
            }

            $item_info->{attempts_left} --;
            my $item = $items->[ $item_info->{position} ];

            $requests_in_progress += 1;

            # кешируем, чтобы дважды не вызывать prepare_request
            $item_info->{request} = $prepare_request ? $prepare_request->($item) : $item
                if !exists $item_info->{request};
            my $request = $item_info->{request};

            my $is_replanned;
            my $replan_sub = sub {
                return if $is_replanned;
                return if $item_info->{attempts_left} < 1;
                push @xqueue, $item_info;
                $req_count ++;
                $is_replanned = 1;
                push @stats, ['retry'];
            };
            
            my $handle; 
            $handle = Yandex::TCPMultiplexor::SocketHandler->new(
                connect    => [$options{host}, $options{port}],
                timeout    => $options{timeout},
                on_prepare => sub { $options{timeout} },
                ( $options{retry_interval} && $item_info->{attempts_left} > 0
                    ? ( timers => { $options{retry_interval} => sub { $replan_sub->(); $process_queue->(); } } )
                    : ()
                ),
                ( map {
                    my $event = $_;
                    "on_$event" => sub {
                        my $process_time = 0 + sprintf "%.3f", Time::HiRes::time() - $item_info->{start_time};
                        push @stats, [$event, time, $process_time];
                        # убираем хендл из активных
                        $item_info->{handles} = [ grep {$_ ne $handle} @{ $item_info->{handles} } ];

                        if ( $item_info->{attempts_left} > 0 ) {
                            $replan_sub->();
                            $next_step->($handle);
                        } elsif (
                            my $error = !@{ $item_info->{handles} }
                                && $options{callbacks}->{"on_$event"}
                                && $options{callbacks}->{"on_$event"}->($item)
                        ) {
                            # прерываем выполнение
                            $handle->destroy();
                            $cv->send($error);
                        } else {
                            $next_step->($handle);
                        }
                    }
                } qw/error timeout eof/
                # эти коллбэки могут вернуть код ошибки, тогда мы прерываем выполнения, возвращая этот код
                ),
            );

            push @handles, $handle;
            push @{ $item_info->{handles} }, $handle;
            $item_info->{start_time} = Time::HiRes::time();
            $send_request->($request, $handle,
                sub {
                    my ($ae_handle, $res) = @_;

                    # не обрабатываем повторно пришедший результат
                    if ( !$item_info->{is_ready} ) {
                        my $time = Time::HiRes::time();
                        my $process_time = 0 + sprintf "%.3f", $time - $item_info->{start_time};
                        eval {
                            # если упал обработчик ответа, считаем, что ответ был с ошибкой (не ставим флаг успеха)
                            $process_response->($item, $res);
                            push @stats, ['success', int($time), $process_time];
                            $item_info->{is_ready} = 1;
                        }
                        or do {
                            push @stats, ['bad_response', int($time), $process_time];
                            if ( $item_info->{attempts_left} < 1 ) {
                                my $bad_cb = $options{callbacks}->{on_bad_response};
                                if ( my $error = $bad_cb && $bad_cb->($item, $res) ) {
                                    $cv->send($error);
                                    return;
                                }
                            }
                            else {
                                $replan_sub->();
                            }
                        };
                    }

                    # если не было ошибки, чистим все хендлы с этим запросом
                    if ( $item_info->{is_ready} ) {
                        my $all_handles = delete $item_info->{handles};
                        $next_step->($_) for @$all_handles
                    }
                    # а если ошибка была, то только обработанный
                    else {
                        $next_step->($handle);
                        $item_info->{handles} = [ grep {$_ ne $handle} @{ $item_info->{handles} } ];
                    }
                }
            );
        }
    };

    # это надо делать, когда мы разобрались с очередным item
    $next_step = sub {
        my ($handle) = @_;
        
        $handle->destroy();
        $items_ready += 1;
        $requests_in_progress -= 1;
        
        if ( $items_ready == $req_count ) {
            # закончили обработку, сообщаем об этом
            $cv->send(0);
        } else {
            $process_queue->();
        }
    };

    # поехали!
    $process_queue->();

    # дожидаемся последнего item'а или ошибки
    my $error = $cv->recv();

    if ($error) {
        # в этот момент некоторые задачи еще могут быть незавершенными
        for my $handle (@handles) {
            $handle->destroy();
        }
    }

    # уничтожаем наши циклические ссылки
    undef $process_queue;
    undef $next_step;

    die $error if $error;

    return \@stats;
}

1;





package Yandex::TCPMultiplexor::SocketHandler;

# ABSTRACT: AnyEvent::Handle с дополнительными таймерами

use base 'AnyEvent::Handle';

use AnyEvent;
use AnyEvent::Handle;


sub new {
    my ($class, %opt) = @_;

    my @timers;
    if ( my $timers = delete $opt{timers} ) {
        while ( my ($after, $t_handler) = each %$timers ) {
            push @timers, AnyEvent->timer( after => $after, cb => $t_handler );
        }
    }
    
    my $self = $class->SUPER::new(%opt);
    $self->{_ya_timers} = \@timers;

    return $self;
}


sub destroy {
    my ($self) = @_;

    delete $self->{_ya_timers};
    $self->SUPER::destroy()  if !$self->destroyed();

    return;
}


1;

