#! /usr/bin/perl

=head1 memory_leak.t

    Тест проверяет, что Yandex::TCPMultiplexor не утекает памятью. 

=cut

use strict;
use Test::More tests => 2;

use Yandex::TCPMultiplexor;
use Yandex::ProcInfo;
use IO::Socket;
use AnyEvent::Socket;
use Encode;
use Data::Dumper;

# количество итераций
our $iterations = 200;
# количество заданий в одном запросе
our $task_cnt = 60;

my $host = '127.0.0.1';
my $port = get_free_tcp_port();

# Запускаем тестовый сервер
my $rec_counter = 0;
my $tcp_server = tcp_server $host, $port, sub {
    my ($fh, $host, $port) = @_;

    return if ($rec_counter++ %5) < 1;  # иногда падаем, чтобы перепослать запрос
    syswrite $fh, "Quite long test response\015\012" x 20;
};

# Делаем запросы и считаем память
my $size0 = int(proc_memory() / 1024 / 1024);
make_requests($host, $port);
my $size1 = int(proc_memory() / 1024 / 1024);

# проверяем рост памяти после первого запроса
ok($size1 - $size0 <= 1, "mem growth after the first request (Mb): $size0 --> $size1");

for my $i (0 .. $iterations){
    make_requests($host, $port);
}
my $sizeN = int(proc_memory() / 1024 / 1024);

# проверяем рост памяти после всех итераций
ok($sizeN - $size1 <= 1, "mem growth after $iterations iterations (Mb): $size1 --> $sizeN");


sub make_requests
{
    my ($host, $port) = @_;

    my $tasks = [];

    for my $i (1 .. $task_cnt){
        push @$tasks, "test " x 50;
    }

    my $resp_counter = 0;
    Yandex::TCPMultiplexor::process_queue( 
        $tasks,
        callbacks => {
            send_request     => sub {
                my $query    = shift;
                my $handle   = shift;
                my $callback = shift;

                $handle->push_write(Encode::encode_utf8($query."\n"));
                $handle->push_read(line => $callback);
            }, 
            process_response => sub {
                # эмулируем ошибочный ответ
                die if ($resp_counter++ %4) == 1;

                return;
            },
            on_error         => sub { return },
            on_timeout       => sub { print STDERR "on_timeout"; },
        },
        host => $host,
        port => $port,
        timeout => 5,
        max_requests => 6,
        num_retries => 5,
    );
}


sub get_free_tcp_port
{
    my (%O) = @_;

    $O{from} ||= 10000;
    $O{to} ||= 11000;
    $O{tries} ||= 10;
    $O{order} ||= 'rnd'; 

    for my $i ( 1 .. $O{tries} ) {
        my $delta = int rand($O{to} - $O{from} + 1);
        my $port = $O{from} + $delta;
        last if $port > $O{to} || $port < $O{from};

        my $socket = IO::Socket::INET->new(
            Proto => 'tcp',
            LocalPort => $port,
            Listen => 5,
            Reuse => 1,
            Blocking => 1,
        );

        return $port if $socket;
    }

    die "can't find free port between $O{from} and $O{to} after $O{tries} attempts";
}

