#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Test::More;

use Encode;
use List::MoreUtils qw/ all /;

use Yandex::TCPMultiplexor;



our $DEBUG = 0;


# количество заданий
my $task_cnt = 30;

my $num_parallel = 5;
my $timeout = 3;
my $resend_timeout = 1,
my $attempts = 2;
my %delay = %TestServer::DELAY = (
    1 => [-2, 1],
    2 => [-2, -1],
    6 => [4, 0],
    7 => [4, 4],
    8 => [-2, -2],

    28 => [2, 4],
    29 => [2, 0],
    30 => [2, 1],
);



my $host = '127.0.0.1';

# init forking tcp-server
my $port = get_free_tcp_port();
my $pid = fork();
die "No forks"  if !defined $pid;
if ( !$pid ) {
    # выходим через 2 минуты, на случай сбоя в основном процессе
    local $SIG{ALRM} = sub { exit };
    alarm 120;

    TestServer->run( port => $port, log_level => 0 );
    die "Test-server bug!\n";
}

sleep 1;



my @reqs = map { "request $_" } (1 .. $task_cnt);
my %resp_count;
my %resp_error;
my %resp_timeout;
my %resp_bad;

make_requests($host, $port, \@reqs);

for my $req_id (1 .. $task_cnt) {
    my $req = "request $req_id";
    my $d = $delay{$req_id};
    my $should_fail = $d && ( all {my $t=$d->[$_]; $t && ($t<0 || $t>$timeout)} (0 .. $attempts-1) );

    if ( $should_fail ) {
        ok !$resp_count{$req} => "$req: failed";
        my $last_delay = $d->[$attempts-1];
        if ( $last_delay == -1 ) {
            is $resp_error{$req}, 1 => "$req: single 'error' callback run";
        }
        elsif ( $last_delay == -2 ) {
            is $resp_bad{$req}, 1 => "$req: single 'bad response' callback run";
        }
        elsif ( $last_delay >= 0 ) {
            is $resp_timeout{$req}, 1 => "$req: single 'timeout' callback run";
        }
    }
    else {
        is $resp_count{$req}, 1 => "$req: got single response";
    }
}



# close server
kill TERM => $pid;
wait;

done_testing();





sub make_requests
{
    my ($host, $port, $reqs) = @_;

    Yandex::TCPMultiplexor::process_queue( 
        $reqs,
        callbacks => {
            send_request     => sub {
                my ($query, $handle, $callback) = @_;
                $handle->push_write( encode utf8 => "$query\n" );
                $handle->push_read(line => $callback);
                verbose($query);
            }, 
            process_response => sub {
                my ($req, $resp) = @_;
                verbose($resp);
                die if $resp =~ /corrupt/x;
                ok( $resp =~ $req, "$req: right response" );
                $resp_count{$req} ++;
                return;
            },
            on_bad_response   => sub {
                my ($req) = @_;
                verbose("$req bad response");
                $resp_bad{$req} ++;
                return;
            },
            on_error   => sub {
                my ($req) = @_;
                verbose("$req error");
                $resp_error{$req} ++;
                return;
            },
            on_timeout => sub {
                my ($req) = @_;
                verbose("$req timeout");
                $resp_timeout{$req} ++;
                return;
            },
        },
        host => $host,
        port => $port,
        timeout => $timeout,
        max_requests => $num_parallel,
        retry_interval => $resend_timeout,
        num_retries => $attempts-1,
    );
}


sub get_free_tcp_port
{
    my (%O) = @_;

    $O{from} ||= 10000;
    $O{to} ||= 11000;
    $O{tries} ||= 10;
    $O{order} ||= 'rnd'; 

    for my $i ( 1 .. $O{tries} ) {
        my $delta = int rand($O{to} - $O{from} + 1);
        my $port = $O{from} + $delta;
        last if $port > $O{to} || $port < $O{from};

        my $socket = IO::Socket::INET->new(
            Proto => 'tcp',
            LocalPort => $port,
            Listen => 5,
            Reuse => 1,
            Blocking => 1,
        );

        return $port if $socket;
    }

    die "can't find free port between $O{from} and $O{to} after $O{tries} attempts";
    return;
}

sub verbose {
    return if !$DEBUG;
    print STDERR shift() . "\n";
    return;
}



package TestServer;

use base 'Net::Server::Fork';

use File::Temp;
use Fcntl ':flock';

use JSON;
use Time::HiRes;

our $filename;
BEGIN { $filename = tmpnam() }

our %DELAY;

sub _get_update_attempt_num {
    my ($id) = @_;

    open my $file, '+>>', $filename;
    flock $file, LOCK_EX;
    sysseek $file, 0, 0;
    sysread $file, my $buf, 1000;

    my $values = decode_json( $buf || '[]' );
    my $result = $values->[$id] || 0;
    $values->[$id] = $result + 1;
    $buf = encode_json( $values );

    sysseek $file, 0, 0;
    truncate $file, 0;
    syswrite $file, $buf;
    
    close $file;

    return $result;
}


sub process_request {
    my ($self) = @_;

    sysread \*STDIN, my $req, 10000;
    chomp $req;

    my ($req_id) = $req =~ /(\d+)/xms;

    my $attempt = _get_update_attempt_num($req_id);

    my $delays = $DELAY{$req_id};
    my $response = "response for $req, attempt $attempt [$$]\x0D\x0A";

    if ( my $delay = $delays && $delays->[$attempt] ) {
        if ( $delay == -1 ) {
            print STDERR "[$$] emulating error for $req (attempt $attempt)\n"  if $main::DEBUG;
            return;
        }
        if ( $delay == -2 ) {
            print STDERR "[$$] bad response for $req (attempt $attempt)\n"  if $main::DEBUG;
            $response = "corrupted $response";
        }
        else {
            print STDERR "[$$] delaying $req (attempt $attempt) for $delay secs\n"  if $main::DEBUG;
            sleep $delay;
        }
    }

    syswrite \*STDOUT, $response;
    return;
}


1;

