package Yandex::Trace;

use strict;
use warnings;
use utf8;
use feature 'state';
use base qw/Exporter/;
use Readonly;
use Carp;
use Scalar::Util qw/weaken isweak/;

use Time::HiRes qw//;
use Yandex::Hostname;
use Yandex::ListUtils qw/xsort/;
use List::MoreUtils qw/any/;
use List::Util qw/sum first/;
use JSON;
use POSIX qw/strftime/;
use Log::Any;
use Yandex::ProcInfo qw/proc_memory/;
use Log::Any::Adapter;

=head1 NAME

Yandex::Trace

=head1 SYNOPSYS

my $trace = Yandex::Trace->new(service => '...', method => '...');

my $profile = Yandex::Trace::new_profile('func_name');
sleep 10;
undef $profile;
undef $trace;

=head1 DESCRIPTION

Порядок наименования функций в new_profile:
1. Имена не должны зависеть от языка или особенностей реализации. Нужно стремиться к тому, чтобы
имя профайла, измеряющего одно и то же из разных языков совпадало.
Например, писать dbtools:exec_sql неправильно - это особенности нашей библиотеки, и в python аналогичная
функция может называться по другому. Правильно будет db:exec
1.1. Если в другом языке уже написан профайлинг - нужно повторять за ним с точностью до буквы.

2. Отдельные части имени разделяются двоеточием
2.1. В начале имени пишется namespace.
2.2. Если замеряется время работы http ручки, то имя создается на основании url:
    $host/api/1.1/client/info => 'client:info'
    /client/$uid/status => 'client:status'
    слеш в url заменяется на двоеточие
2.3 Если имя функции содержит в себе неймспейс - упрощаем имя функции:
Вместо mymodule:mymodule_get пишем mymodule:get

3. При профилировании кода Директа или прочих сервисов, неймспейс выбирается из
имени модуля/класса/скрипта:
mod_export:get_campaigns_for_moderate
bsActiveOrders:process_orders

=cut

our @EXPORT_OK = qw/
    current_trace
    $INDEX
    $PROFILE_INDEX
/;

=head2 _hostname

Кеширующая версия hostfqdn()
Предполагается что hostfqdn может упасть, поэтому //=

=cut

sub _hostname
{
    state $hostname;
    $hostname //= Yandex::Hostname::hostfqdn();
    return $hostname;
}

Readonly my $FORMAT_ID => 3;

# позиции в _data_for_log
Readonly our $INDEX => {
    FORMAT => 0,
    DATE_TIME => 1,
    HOSTNAME => 2,
    PID => 3,
    SERVICE => 4,
    METHOD => 5,
    TAGS => 6,
    TRACE_ID => 7,
    PARENT_ID => 8,
    SPAN_ID => 9,
    CHUNK_NUM => 10,
    LAST_CHUNK => 11,
    SPAN_ELA => 12,
    SAMPLERATE => 13,
    DATA => 14,
};

# позиции внутри секции profile
Readonly our $PROFILE_INDEX => {
    FUNC => 0,
    TAGS => 1,
    ALL_ELA => 2,
    CHILDS_ELA => 3,
    CALLS => 4,
    OBJ_NUM => 5,
};

Readonly my $HTTP_HEADER_NAME => 'X-Yandex-Trace';
Readonly my $ENV_VAR_NAME => 'YANDEX_TRACE';

# если менее чем REST_THRESHOLD покрыто профайлами, добавляем профайл 'rest'
Readonly my $REST_THRESHOLD => 0.95;

# параметры для настройки логгирования, передаются в Log::Any::Adapter->set
our $LOG_ANY_ADAPTER //= 'YandexLog';
our $LOG_OPTIONS //= {
    log_file_name => 'trace.log',
    no_date => 1,
    date_suf => '%Y%m%d',
};


=head2 $TRACE

Текущий трейс, слабая ссылка на объект
Существует в единственном экземпляре на процесс
После форка в дочернем процессе необходимо обновить:
$trace = current_trace()

=cut

my $TRACE;

=head2 current_trace

    Returns current trace object.

    my $current_trace_start_time = current_trace()->{start_time};

=cut

sub current_trace
{
    my (%opt) = @_;
    if ($TRACE && !$opt{no_flush}) {
        $TRACE->smart_flush();
    }
    return $TRACE;
}

=head2 current_span_id

    Текущний span_id (из $TRACE) служит для reqid
    https://st.yandex-team.ru/DIRECT-43679

=cut

sub current_span_id { $TRACE ? $TRACE->{span_id} : undef }

=head2 current_trace_id

    Текущний trace_id (из $TRACE)

=cut

sub current_trace_id { $TRACE ? $TRACE->{trace_id} : undef }
=head2 new_from_http_headers

my $trace = new_from_http_headers($r->headers, %trace_params)

Парсит http заголовок X-Yandex-Trace и создает новый объект трейса

Первый параметр должен быть типа HTTP::Header
Остальные параметры те же что и для new()

=cut

sub new_from_http_headers
{
    my $headers = shift;
    return new_from_str(scalar($headers->header($HTTP_HEADER_NAME)), @_);
}

=head2 new_from_env

Создать объект trace, проинициализировать родительские id значениями
из переменной окружения

Параметры те же что и для new()

=cut

sub new_from_env
{
    return new_from_str($ENV{$ENV_VAR_NAME}, @_);
}

=head2 new_from_str

Создать объект trace, проинициализировать родительские id значениями
из строки

Первый параметр - строка
Остальные параметры те же что и для new()

=cut

sub new_from_str
{
    my ($str, %opt) = @_;
    $str //= '';
    my ($trace_id, $span_id, $parent_id) = split /,/, $str;
    if (any { length && !/^\d+$/ } ($trace_id, $parent_id, $span_id)) {
        warn "invalid values in ids string: '$str'";
    }
    return Yandex::Trace->new(%opt, trace_id => $trace_id, parent_id => $parent_id, span_id => $span_id);
}

=head2 restart

class method to restart trace and change some params in new object
usage:
    my $trace = Yandex::Trace->new(...);

    Yandex::Trace::restart(\$trace, %options);

    %options are the same as for new()

=cut

sub restart
{
    my $trace_ref = shift;
    if (ref $trace_ref ne 'REF') {
        croak "ref to trace object required, got '".(ref $trace_ref)."'";
    }
    $$trace_ref = $$trace_ref->_clone(@_);
}

=head1 INTERNALS

Внутреннее устройство Yandex::Trace

profile:
    Вложеный хеш, в котором хранится сгруппированная статистика по вызовам:
        $trace->{profile}->{$func_name}->{$tags} = {
            func => $func_name, # имя функции
            tags => $tags,      # теги
            obj_num => $n,      # количество объектов
            calls => $calls,    # количество вызовов
            childs_ela => $n,   # время, затраченное дочерними профайлами
            all_ela => $n,      # время, затраченное даным вызовом
        }

profile_calls:
    Массив объектов Yandex::Trace::Call (слабые ссылки)
    Нужен для получения родительского профайла при создании нового

Yandex::Trace::Call
    Объект, который создается вызовом new_profile. В объект передается слабая
    ссылка на хеш из $trace->{profile}->{$func}->{$tags}. В момент разрушения
    Y::T::Call обновляет данные о вызове внутри этого хеша (таким образом,
    внутри $trace)

=cut

=head2 new

Параметры именованные:
обязательные:
    method => имя метода
    service => имя сервиса

необязательные:
    tags => 'tag1,tag2' -- теги, строка через запятую
    annotations => '...' -- аннотации, строка через запятую

    trace_id  => N
    span_id   => N
    parent_id => N -- для установки ссылки на родилельский trace

    flush_timeout => N (60) -- время, после которого накопленная
        статистика автоматически сбросится в лог

=cut

sub new
{
    my ($class, %self) = @_;
    
    Log::Any::Adapter->set({ category => 'Yandex::Trace' }, $LOG_ANY_ADAPTER, %$LOG_OPTIONS);

    # print STDERR "new in ($$)\n";

    $self{method} or croak "name required";
    $self{service} or croak "service required";
    $self{tags} //= '';
    $self{start_time} //= Time::HiRes::time();
    $self{pid} = $$;
    $self{hostname} = _hostname;
    if (!$self{trace_id}) {
        # в корневом трейсе span_id равен trace_id
        $self{trace_id} = $self{span_id} = generate_traceid(),
    }
    else {
        if (!$self{span_id}) {
            croak "trace_id given but span_id missing";
        }
    }
    $self{parent_id} //= 0;
    $self{chunk_num} //= 1;
    $self{samplerate} //= 1;
    $self{profile} = {};
    $self{services} = [];
    $self{services_data} = [];
    $self{profile_calls} = [];
    $self{marks} //= [];
    $self{annotations} //= [];
    $self{times_0} = [times()];
    $self{mem_0} = proc_memory();
    $self{log} = Log::Any->get_logger(category => 'Yandex::Trace');

    $self{trace_start_time} = Time::HiRes::time();

    $self{flush_timeout} //= 60;

    my $self = bless \%self, $class;
    $TRACE = $self;
    weaken $TRACE;
    return $self;
}

# Class::Accessor in a nutshell
for my $field (qw/method service tags trace_id parent_id span_id annotations/) {
    no strict 'refs';
    *{"Yandex::Trace::$field"} = sub {
        my $self = shift;
        if (!$self) {
            # Y::T::method()
            $self = $TRACE;
        }
        elsif (ref $self eq '') {
            # Y::T::method(42)
            croak "setter as method class not allowed";
        }
        if (@_) {
            $self->{$field} = $_[0];
        }
        return $self->{$field};
    };
}

=head2 _clone

Создать новый объект на основе текущего
Именованные параметры - те же что у new()

=cut

sub _clone
{
    my $self = shift;
    my %opt = @_;
    my %copy = ();
    # print STDERR "clone self($self->{pid}) in ($$)\n";
    my %skip = map { $_ => 1 } qw/pid profile profile_calls services_data trace_id parent_id span_id start_time/;
    $copy{$_} = $self->{$_} for grep { !$skip{$_} } keys %$self;
    return Yandex::Trace->new(%copy, %opt);
}

=head2 generate_traceid

Создать уникальное 63-битное (знаковое, положительное) целое

=cut

sub generate_traceid
{
    return (
        (int(Time::HiRes::time()*256) & 0x7FFF_FFFF) << 32
        |
        ($$ & 0xFFFF) << 16
        |
        int(rand(0xFFFF))
        );
}

=head2 _log_time_str

Вернуть текущее время в UTC в формате yyyy-mm-dd hh:mm:ss.ff, где ff - дробная часть секунды с точностью до 6 знака

Параметр - unix time, в виде числа с плавающей точкой

=cut

sub _log_time_str
{
    my $self = shift;
    my $now = shift;
    my $ms = sprintf "%.6f", $now;
    $ms =~ s!\d+\.!!;
    return strftime("%F %T.$ms", gmtime($now));
}

=head2 _data_for_log

Вернуть данные для записи в лог

=cut

sub _data_for_log
{
    my $self = shift;
    my $end_time = Time::HiRes::time();
    my $span_ela = $end_time - $self->{start_time};
    my $times = [times()];
    my $mem = proc_memory();
    my $times_h = {
        ela => $span_ela,
        cu  => $times->[0] - $self->{times_0}->[0],
        cs  => $times->[1] - $self->{times_0}->[1],
        mem => int(($mem - $self->{mem_0}) / 1024 / 1024),
    };
    my $line = [
        $FORMAT_ID + 0,
        $self->_log_time_str($end_time),
        $self->{hostname},
        $self->{pid} + 0,
        $self->{service},
        $self->{method},
        $self->{tags},
        $self->{trace_id} + 0, 
        $self->{parent_id} + 0,
        $self->{span_id} + 0,
        $self->{chunk_num} + 0,
        JSON::false,
        $end_time - $self->{trace_start_time},
        $self->{samplerate} + 0,
        {
            times => $times_h,
            profile => $self->_profile_data(),
            services => $self->{services_data}, 
            marks => $self->{marks},
            annotations => $self->{annotations},
        }
    ];
    my @funcs = keys %{$self->{profile}};

    # если профайл завершился - смотрим на его all_ela;
    # для незавершившихся профайлов - считаем от их start_time до текущего времени
    my $profiled_time =
        sum 0,
        map { ($_->{all_ela} // ($end_time - $_->{start_time})) - $_->{childs_ela} }
        map { values %{$self->{profile}->{$_}} } @funcs;
    my $rest_time = $span_ela - $profiled_time;
    if ($span_ela > 0 && $profiled_time / $span_ela < $REST_THRESHOLD) {
        # если менее чем REST_THRESHOLD% кода покрыто профайлингом - добавляем rest
        push @{$line->[$INDEX->{DATA}]->{profile}}, ['rest', '', $rest_time, 0, 1, 0];
    }
    return $line;
}

=head2 new_profile

my $t = current_trace()->new_profile('json:decode', obj_num => $count, tags => '...');
json_decode($str);
undef $t;

Positional arguments:
    $name - name of the function

Named arguments:
    tags => 'tag1,tag2' - tags, used to group functions
    obj_num => number of object, processed by func
    no_flush => 0|1 - не пытаться сделать flush()

Соглашение о наименовании смотри в начале файла

=cut

sub new_profile
{
    unless (defined wantarray) {
        carp "new_profile in void context is useless";
    }
    
    my $self = (ref $_[0] eq 'Yandex::Trace' ? shift : undef);

    my $func = shift;
    croak "func required" unless $func;
    
    my (%opt) = @_;
    
    if ($self) {
        if (!$opt{no_flush}) {
            $self->smart_flush();
        }
    }
    else {
        $self = current_trace(no_flush => $opt{no_flush});
    }
    
    # группируем профили по имени (func) и тегам
    # ищем существующий или создаем новый хеш
    my $call_profile = $self->{profile}->{$func}->{$opt{tags}//''} //= {
        func => $func,
        tags => $opt{tags} // '',
    };
    $call_profile->{obj_num} += $opt{obj_num} // 0;
    $call_profile->{calls}++;
    $call_profile->{childs_ela} //= 0;
    my $parent = defined $self->{profile_calls} ? $self->{profile_calls}->[-1] : undef;
    my $call = Yandex::Trace::Call->new(
        func => $func,
        %opt,
        profile => $call_profile,
        trace => $self,
        parent => $parent,
    );
    weaken $call->{profile};
    push @{$self->{profile_calls}}, $call;
    weaken $self->{profile_calls}->[-1];
    return $call;
}

=head2 new_service

Профилирование внешнего сервиса

my $service = Yandex::Trace::new_service('serv_name', 'method_name');
my @ids     = $service->get_service_ids(); # (trace, span, parent)
call_service_method();
undef $service;

См. также документацию к Yandex::Trace::Service->new

=cut

sub new_service
{
    unless (defined wantarray) {
        carp "new_service in void context is useless";
    }
    
    my $self = (ref $_[0] eq 'Yandex::Trace' ? shift : current_trace());
    $self->smart_flush();

    my $service_name = shift;
    my $method_name = shift;
    
    my (%opt) = @_;

    my $service = Yandex::Trace::Service->new(service => $service_name, method => $method_name, %opt);

    return $service;
}

=head2 profile_data

Данные профайлинга в формате для записи в лог

=cut

sub _profile_data
{
    my $self = shift;
    my @funcs = keys %{$self->{profile}};
    return [
        map { [ $_->{func}, $_->{tags}//'', $_->{all_ela}, $_->{childs_ela}, $_->{calls}, $_->{obj_num} ] } 
        grep { $_->{all_ela} } # только завершенные
        map { values %{$self->{profile}->{$_}} } @funcs
    ];
}

=head2 marks

$trace->mark('something happened');
Добавляет в marks отметку о каком-то событии, с указанием относительного времени
(относительно времени старта трейса)

=cut

sub mark
{
    my $self = shift;
    my $mark = shift;
    push @{$self->{marks}}, [ Time::HiRes::time() - $self->{trace_start_time}, $mark ];
}

=head2 annotate

$trace->annotate(version => 'v5');

Добавить в аннотации пару ключ-значение

=cut

sub annotate
{
    my ($self, $key, $value) = @_;
    unless (defined $key && defined $value) {
        croak "both key and value required";
    }
    push @{$self->{annotations}}, [ $key => $value ];
}

=head2 flush

Вручную сбросить накопленную статистику в лог и очистить статистику

=cut

sub flush
{
    my $self = shift;
    
    return if $self->{pid} != $$;

    $self->{log}->info(encode_json($self->_data_for_log()));
    $self->{start_time} = Time::HiRes::time();
    # удаляем завершенные профайлы
    for my $func (keys %{$self->{profile}}) {
        for my $tag (keys %{$self->{profile}->{$func}}) {
            if ($self->{profile}->{$func}->{$tag}->{all_ela}) {
                delete $self->{profile}->{$func}->{$tag};
            }
        }
        if (scalar keys %{$self->{profile}->{$func}} == 0) {
            delete $self->{profile}->{$func};
        }
    }
    $self->{services_data} = [];
    $self->{marks} = [];
    $self->{chunk_num}++;
    $self->{times_0} = [times];
    $self->{mem_0} = proc_memory();
}

=head2 smart_flush

Сбросить статистику в лог если с момента последнего сброса прошло
более flush_timeout секунд.

Вызывается автоматически в new_profile и current_trace, но можно
вызвать самостоятельно, например внутри длинного цикла

=cut

sub smart_flush
{
    my $self = shift;
    my $timeout = $self->{flush_timeout};
    if (Time::HiRes::time() - $self->{start_time} >= $self->{flush_timeout}) {
        $self->flush();
    }
}

sub DESTROY
{
    my $self = shift;
    # print STDERR "dies for pid = $self->{pid} in $$\n";
    return unless $self->{pid} == $$;
    my $line = $self->_data_for_log();
    $line->[$INDEX->{LAST_CHUNK}] = JSON::true;
    $self->{log}->info(encode_json($line));
}

=head2 profile_readable

my $profile = from_json($line)->[14]->{profile}
my $profile_as_array_of_hashes_sorted = Yandex::Trace::profile_readable($profile);

Функция форматирует список профайлов в том виде, как они записаны в логе, в формат пригодный для чтения человеком
Также, делается сортировка по all_ela

=cut

sub profile_readable
{
    my $profile = shift;
    return [] unless $profile && @$profile;
    $profile->[0]->[2] += 0;
    return [ map { _pp_profile($_) } xsort { \$_->[2] } @$profile ];
}

=head2 _pp_profile

=cut

sub _pp_profile
{
    my $ar = shift;
    return { map { lc $_ => $ar->[ $PROFILE_INDEX->{$_} ] } keys %$PROFILE_INDEX };
}

package Yandex::Trace::Service {

use strict;
use warnings;
use utf8;
use Carp;
use Time::HiRes qw//;
use Scalar::Util qw/weaken/;

=head2 

3. Порождение дочернего спана: у спана (T, S, P, TTL) есть операция create_child, которая создаёт кортеж: (T', S', P', TTL')
Данные в кортеже такие:
T' = T
S' — новое число
P' = S
TTL' = TTL - 1 или 0 (меньше нуля быть не может)

Явно создавать Yandex::Trace::Service не нужно, для этого есть Yandex::Trace::new_service()
Пример использования:
my $trace = Yandex::Trace->new(...);
my $service = Yandex::Trace::new_service('some:service', 'method_name');
my $res = http_get($SOME_SERVICE_URL, headers => $service->http_headers);
my ($trace_id, $span_id, $parent_id, $ttl) = $service->get_service_ids()
undef $service;

Сервисы отличаются от profile тем, что никогда не группируются, каждый объект - отдельная запись
в профайл логе.
Их назначение - связь профилирования в разных подсистемах через parent_id -> span_id

Пояснение о trace_id, span_id, parent_id:
Для самого первого трейса trace_id = span_id = generate_traceid(), parent_id = 0
Для дочернего трейса: C.trace_id = P.trace_id, C.span_id = generate_traceid(), C.parent_id = P.span_id
То есть, trace_id не меняется для всех дочерних сервисов, parent_id указывает на span_id родительского
трейса, span_id всегда генерируется новый

=cut

sub new
{
    my ($class, %self) = @_;
    croak "service required" unless $self{service};
    croak "method required" unless $self{method};

    $self{trace_id} = $TRACE->{trace_id};
    $self{span_id} = $TRACE->generate_traceid();
    $self{parent_id} = $TRACE->{span_id};

    $self{pid} = $$;

    $self{parent_trace} = $TRACE;
    weaken $self{parent_trace};

    $self{start_time} = Time::HiRes::time();

    return bless \%self, $class;
}

=head2 http_headers

Вернуть ссылку на хеш, который можно использовать для выставления
http заголовков при передаче трейса во внешний сервис

http_get($url, headers => $service->http_headers)

my %services = ();
http_parallel_request(GET => \%req, prepare_callback => sub {
        my ($id, $params, $task) = @_;
        # TODO: доработать Yandex::HTTP, чтобы в prepare_callback передавался id
        my $s = Yandex::Trace::new_service('bs', 'bsrank');
        $services{$id} = $s;
        $params->{headers} = $s->http_headers();
    },
    callback => sub {
        my ($id, $res) = @_;
        delete $services{$id};
    },
    );

=cut

sub http_headers
{
    my $self = shift;
    return {
        $HTTP_HEADER_NAME => (join ',', $self->get_service_ids())
    };
}

=head2 get_service_ids

Вернуть массив из trace_id, span_id, parent_id

=cut

sub get_service_ids
{
    my $self = shift;
    return @{$self}{qw/trace_id span_id parent_id/};
}

sub DESTROY
{
    my $self = shift;
    
    return unless $self->{pid} == $$;

    my $end_time = Time::HiRes::time();
    $self->{ela} = $end_time - $self->{start_time};
    push @{$self->{parent_trace}->{services_data}}, [
        $self->{service},
        $self->{method},
        $self->{span_id},
        # rel_client_send - относительное время начала сервиса
        $self->{start_time} - $self->{parent_trace}->{trace_start_time},
        $self->{ela},
    ];
}

};


package Yandex::Trace::Call {

=head1 DESCRIPTION

Объект, который возвращается из new_profile

Хранит в себе статистику про текущий вызов

При создании в объект передается ссылка на хеш profile => {}
При уничтожении в этот объект записывается статистика

=cut

use strict;
use warnings;
use utf8;
use Carp;
use Scalar::Util qw/weaken isweak/;
use Time::HiRes qw//;

sub new
{
    my ($class, %self) = @_;
    croak "func required" unless $self{func};
    die "profile missing" unless $self{profile};
    die "trace missing" unless $self{trace};
    weaken $self{trace};
    weaken $self{parent};
    if ($self{parent}) {
        $self{_has_parent} = 1;
    }
    $self{pid} = $$;
    $self{start_time} = Time::HiRes::time();
    $self{profile}->{start_time} = $self{start_time};
    return bless \%self, $class;
}

=head2 obj_num

$profile->obj_num(scalar @items);
print $profile->obj_num();

Прочитать или установить значение obj_num 

=cut

sub obj_num
{
    my $self = shift;
    if (@_) {
        $self->{profile}->{obj_num} = shift;
    }
    return $self->{profile}->{obj_num};
}

sub DESTROY
{
    my $self = shift;
    # print STDERR "DESTROY $self->{func}\n";
    my $end = Time::HiRes::time();
    my $ela = $end - $self->{start_time};
    $self->{profile}->{all_ela} += $ela;
    if ($self->{parent}) {
        $self->{parent}->{profile}->{childs_ela} += $ela;
    }
    else {
        if ($self->{_has_parent}) {
            warn "parent profile destructed before the child ($self->{func})";
        }
    }

    if ($self->{trace}->{profile_calls} && ($self->{trace}->{profile_calls}->[-1]//'') eq $self) {
        # в нормальном случае (когда нет пересекающихся профайлов) достаточно выкинуть последний профайл
        pop @{$self->{trace}->{profile_calls}};
    }
    else {
        $self->{trace}->{profile_calls} = [ grep { defined } @{$self->{trace}->{profile_calls}//[]} ];
        for (@{$self->{trace}->{profile_calls}}) {
            if (!isweak($_)) {
                weaken($_);
            }
        }
    }
}

};

1;

