#!/usr/bin/perl

use strict;
use warnings;
use Test::More; 
use Test::Deep;
use Test::MockTime qw(:all);
use Log::Any::Test;
use Log::Any qw/$log/;
use List::MoreUtils qw/uniq/;
use JSON;
use Path::Tiny;

use lib::abs '../lib';

do(lib::abs::path('.').'/_test_common.pl');

BEGIN { use_ok('Yandex::Trace', qw/current_trace $INDEX $PROFILE_INDEX/) };

mock_hires();
set_fixed_time(time);

{
my $trace = Yandex::Trace->new(service => 'test', method => 'service');

my $s = Yandex::Trace::new_service('test_s', 'sleep');
test_sleep(5);
undef $s;

$trace->flush();

my $r = get_last_log();

ok(scalar @{$r->[14]->{services}} == 1, "correct num of services");
ok($r->[14]->{services}->[0]->[0] eq 'test_s', "correct service name");
}

{
my $trace = Yandex::Trace->new(service => 'test', method => 'service');

my $s = Yandex::Trace::new_service('test_s1', 'sleep');
test_sleep(5);
undef $s;

my $s2 = Yandex::Trace::new_service('test_s2', 'sleep');
test_sleep(12);

ok($s2->{trace_id} == $trace->{trace_id}, "trace_id => trace_id");
ok($s2->{parent_id} == $trace->{span_id}, "parent => span");

undef $s2;

$trace->flush();

my $r = get_last_log();

ok(scalar @{$r->[14]->{services}} == 2, "correct num of services");
ok($r->[14]->{services}->[0]->[0] eq 'test_s1', "correct service name (0)");
ok($r->[14]->{services}->[1]->[0] eq 'test_s2', "correct service name (1)");

ok($r->[14]->{services}->[0]->[4] == 5,  "correct service ela (0)");
ok($r->[14]->{services}->[1]->[4] == 12, "correct service ela (1)");

}

{
my $trace = Yandex::Trace->new(service => 'test', method => 'test');
$trace->span_id(42);
my $s = Yandex::Trace::new_service('test', 'fork');
local $ENV{YANDEX_TRACE} = join ',', $s->get_service_ids();
my $cmd = lib::abs::path('.') . "/script/example.pl";
my $log_file = `$cmd`;
chomp $log_file;
my $log = path($log_file)->slurp_utf8();
my ($date, $time, $json) = split /\s+/, $log, 3;
my $log_data = from_json($json);


my ($child_trace_id, $child_parent_id, $child_span_id) = @$log_data[7,8,9];
ok($child_trace_id == $trace->trace_id, "child trace_id == parent trace_id");
ok($child_parent_id == $trace->span_id, "child parent_id == parent span_id");
ok($child_span_id != $trace->span_id && $child_span_id != $trace->trace_id, "child span id is uniq");

}


# одинаковое имя сервиса и метода несколько раз подряд
{
my $trace = Yandex::Trace->new(service => 'test', method => 'test');
my (@tids, @sids, @pids); # trace, span, parent

for (1.. 5) {
    my $s = Yandex::Trace::new_service('s', 'm');
    test_sleep(1);
    my ($t, $sid, $p) = $s->get_service_ids();
    push @tids, $t;
    push @sids, $sid;
    push @pids, $p;
    undef $s;
}
undef $trace;
my $r = get_last_log();
my $services = $r->[14]->{services};
# сервисы не группируются
ok(scalar @$services == 5, "same service and method name");
# trace_id не меняется между сервисами
ok(scalar(uniq(@tids)) == 1, "all trace_ids are the same");
# span_id создается новый перед каждым запуском
ok(scalar(uniq(@sids)) == 5, "all sids different");
# parent_id всегда равен trace_id
cmp_deeply(\@tids, \@pids, "tids == pids");
}

# одинаковое имя сервиса, разные методы несколько раз подряд
{
my $trace = Yandex::Trace->new(service => 'test', method => 'test');
my (@tids, @sids, @pids); # trace, span, parent

for (1.. 5) {
    my $s = Yandex::Trace::new_service('s', 'm_'.$_);
    test_sleep(1);
    my ($t, $sid, $p) = $s->get_service_ids();
    push @tids, $t;
    push @sids, $sid;
    push @pids, $p;
    undef $s;
}
undef $trace;
my $r = get_last_log();
my $services = $r->[14]->{services};
ok(scalar @$services == 5, "same service, different method");
ok(scalar(uniq(@tids)) == 1, "all trace_ids are the same");
ok(scalar(uniq(@sids)) == 5, "all sids different");
cmp_deeply(\@tids, \@pids, "tids == pids");
}

# разные сервисы, разные методы несколько раз подряд
{
my $trace = Yandex::Trace->new(service => 'test', method => 'test');
my (@tids, @sids, @pids); # trace, span, parent

for (1.. 5) {
    my $s = Yandex::Trace::new_service('s_'.$_, 'm_'.$_);
    test_sleep(1);
    my ($t, $sid, $p) = $s->get_service_ids();
    push @tids, $t;
    push @sids, $sid;
    push @pids, $p;
    undef $s;
}
undef $trace;
my $r = get_last_log();
my $services = $r->[14]->{services};
ok(scalar @$services == 5, "different service, different method");
ok(scalar(uniq(@tids)) == 1, "all trace_ids are the same");
ok(scalar(uniq(@sids)) == 5, "all sids different");
cmp_deeply(\@tids, \@pids, "tids == pids");
}

{
my $trace = Yandex::Trace->new(service => 'test', method => 'test');

my $s1 = Yandex::Trace::new_service('test', 's1');
my $s2 = Yandex::Trace::new_service('test', 's2');
test_sleep(1);
undef $s2;
undef $s1;

$trace->flush();
my $services = get_last_log()->[14]->{services};
ok(scalar @$services == 2, "new 1; new 2; undef 2; undef 1");

$s1 = Yandex::Trace::new_service('test', 's1');
$s2 = Yandex::Trace::new_service('test', 's2');
test_sleep(1);
undef $s1;
undef $s2;

$trace->flush();
$services = get_last_log()->[14]->{services};
ok(scalar @$services == 2, "new 1; new 2; undef 1; undef 2");
}

{
my $trace = Yandex::Trace->new(service => 'test', method => 'test');

my $s1 = Yandex::Trace::new_service('test', 's1');
test_sleep(1);
}
my $services = get_last_log()->[14]->{services};
ok(scalar @$services == 1, "service destroyed when goes out of scope");

{
my $trace = Yandex::Trace->new(service => 'test', method => 'test');

my $s1 = Yandex::Trace::new_service('test', 's1');
my $h = $s1->http_headers();
ok(ref $h eq 'HASH', "http headesr is a hash");
ok(scalar keys %$h == 1, "correct number of headers");
my ($h_name, $h_value) = %$h;
ok($h_value =~ /\d+,\d+,\d+/, "header value is 3 numbers, comma-separated");
}

done_testing();
