#!/usr/bin/perl

use strict;
use warnings;

use utf8;
use open ':std', ':encoding(UTF-8)';

=head1 SYNOPSIS

  get-tvm2-ticket.pl --src-id 2000389 --dst-id 222 --secret-path /etc/direct-tokens/tvm2_direct-scripts
  get-tvm2-ticket.pl --src-id 2000389 --dst-id 222 --secret-path /etc/direct-tokens/tvm2_direct-scripts --cache-path /tmp/tvm2_ticket_direct-scripts2blackbox

=head1 DESCRIPTION

  Получить TVM-тикет для запроса из приложения, заданного параметром --src-id, в приложение, заданное параметром --dst-id.
  Есть возможность закэшировать тикет в файл, указанный в параметре --cache-path.

=cut

use Getopt::Long;

use Yandex::TVM2;

my ($dst_id, $cache_path);
GetOptions(
    'src-id=i' => \$Yandex::TVM2::APP_ID,
    'dst-id=i' => \$dst_id,
    'secret-path=s' => \$Yandex::TVM2::SECRET_PATH,
    'cache-path=s' => \$cache_path,
    'h|help' => sub { system "podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2"; exit 1 },
) || die "can't parse options, stop\n";
die "error: src-id is missing\n" unless $Yandex::TVM2::APP_ID;
die "error: dst-id is missing\n" unless $dst_id;
die "error: secret-path is missing\n" unless $Yandex::TVM2::SECRET_PATH;

my $ticket;
if ($cache_path) {
    my @stat = stat($cache_path);
    if (@stat && time() - $stat[9] < $Yandex::TVM2::CACHE_TTL) {
        open my $fh, "<", $cache_path or warn "Can't open $cache_path for reading: $!";
        $ticket = do { local $/; <$fh> } if $fh;
    }
}
if (!$ticket) {
    $ticket = Yandex::TVM2::get_ticket($dst_id);
    if ($cache_path) {
        open my $fh, ">", $cache_path or warn "Can't open $cache_path for writing: $!";
        print $fh $ticket;
    }
}
print $ticket;
