package Yandex::TVM2;

=head1 NAME

    Yandex::TVM2
    Библиотека для получения tvm2 тикетов.
    Тикеты кэшируются.

=head1 DESCRIPTION

    Для работы необходимо зарегистрированное приложение в tvm2.
    Его id записать в переменную $id в функции _get_new_tvm2_tickets
    Секрет записать в файл и указать путь к файлу в переменной $SECRET_PATH
    Подробнее здесь https://wiki.yandex-team.ru/passport/tvm2/quickstart/

=cut

use strict;
use warnings;

use feature 'state';

use Digest::SHA qw(hmac_sha256_base64);
use MIME::Base64;
use JSON qw( decode_json );

use Yandex::HTTP;
use Yandex::Trace;

our $APP_ID;
our $SECRET_PATH;
our $API_URL //= 'https://tvm-api.yandex.net/2/ticket';
our $REQUEST_TIMEOUT //= 5;
our $CACHE_TTL //= 50 * 60;     # секунд

my %Cache;
my %SecretByApp;

=head2 _sign($secret, $ts, $dst_client_id, [$scopes])

    Функция переписана из оригинальной библиотеки tvm2ticket.
    Возвращает подпись, необходимую для валидации запроса в tvm2.
    $dst_client_id - строка из id необходимых приложений, разделенных запятыми. Без пробелов.
    По умолчанию $scopes равен пустой строке

=cut

sub _sign {
    my ($secret, $ts, $dst_client_id, $scopes) = @_;
    unless (defined $scopes) {
        $scopes = '';
    }
    my $DELIM = '|';
    my $data = join($DELIM, $ts, $dst_client_id, $scopes).$DELIM;
    $secret =~ s/\-/\+/g;
    $secret =~ s!\_!\/!g;
    my $s = hmac_sha256_base64($data, decode_base64($secret));
    # TVM использует "URL and Filename safe" алфавит (см. п. 4 в rfc3548), в котором другие 62 и 63 символы
    $s =~ tr!+/!-_!;
    return $s
}


=head2 _read_tvm2_secret()

    Читает секрет из файла. В случае неудачи генерирует исключение.
    Путь к файлу указан в $SECRET_PATH

=cut

sub _read_tvm2_secret {
    unless (defined $SECRET_PATH) {
        die "undefined path to secret file"
    }
    open SECRET_FILE, '<', $SECRET_PATH or die "Cannot open file ".$SECRET_PATH.": $!\n";
    my $Secret = <SECRET_FILE>;
    unless (defined $Secret) {
        die "In file ".$SECRET_PATH.": Secret not found";
    }
    chomp $Secret;
    close(SECRET_FILE);
    $SecretByApp{$APP_ID} = $Secret;
}

=head2 _get_parsed_json_response_from_tvm2($dst_arr, [$scopes])

    Ходит за тикетами в tvm-api.
    На вход получает ссылку на массив id нужных приложений.
    Возвращает распарсеный json-ответ.
    В случае неудачи генерирует исключение.

=cut

sub _get_parsed_json_response_from_tvm2 {
    my ($dst_arr, $scopes) = @_;
    my $dst = join(',', @$dst_arr);
    unless (defined $APP_ID) {
        die "_get_parsed_json_response_from_tvm2: Error: tvm2 application id not set";
    }
    unless (defined $SecretByApp{$APP_ID}) {
        _read_tvm2_secret;
    }
    my $ts = time;
    my $response = Yandex::HTTP::http_fetch('POST', $API_URL, {
            'grant_type' => 'client_credentials',
            'src' => $APP_ID,
            'dst' => $dst,
            'ts' => $ts,
            'sign' => _sign($SecretByApp{$APP_ID}, $ts, $dst, $scopes)
        }, timeout => $REQUEST_TIMEOUT, ipv6_prefer => 1);
    my $url = $API_URL;
    my $parsed_json = eval{decode_json($response)} or die "_get_parsed_json_response_from_tvm2: Error: malformed json";
    return $parsed_json;
}

=head2 _get_new_tvm2_tickets($dst_arr, [$scopes])

    Обрабатывает ответ tvm-api.
    На вход получает ссылку на массив id нужных приложений.
    Возвращает hash tvm2 тикетов.
    В случае неудачи генерирует исключение.

=head2 SYNOPSIS

    my $new_tickets = _get_new_tvm2_tickets(['1', '2', '3'], '');
    my $ticket_for1 = $new_tickets->{'1'};

=cut

sub _get_new_tvm2_tickets {
    my ($app_ids, $scopes) = @_;
    $scopes = '' unless defined $scopes;
    my $parsed_json = _get_parsed_json_response_from_tvm2($app_ids, $scopes);
    my %tickets;
    my @bad_resp; # массив описаний ошибок для id у которых не удалось получить тикет
    foreach my $href ($parsed_json) {
        for my $dst_id (keys %$href) {
            if (exists($href->{$dst_id}{"ticket"})) {
                $Cache{$APP_ID}{$dst_id}{$scopes} = [time, $href->{$dst_id}{"ticket"}];
                $tickets{$dst_id} = $href->{$dst_id}{"ticket"};
            } else {
                push @bad_resp, $dst_id.": ".$href->{$dst_id}{"error"};
            }
        }
    }
    for my $app_id (@$app_ids) {
        if (not exists($tickets{$app_id})) {
            push @bad_resp, $app_id.": not found in tvm2 response";
        }
    }
    if (@bad_resp != 0) {
        die "_get_new_tvm2_tickets: Error:\n".join("\n", @bad_resp);
    }
    return \%tickets;
}


=head2 get_tickets($ids_array, [$params])

    Принимает массив id приложений для которых нужны тикеты и hash с параметрами.
    Возможные параметры: 
        cache_enabled - если имеет значение false, то не ищем тикеты в кэше.
                        Если true или неопределен, то сначала ищем в кэше, если не нашли идем в сервис tvm2 по сети
        scopes - список прав
    Возвращает hash tvm2 тикетов для всех id
    Полученные тикеты кэшируются
    Если запрос завершился неудачно либо не смогли получить тикет для одно из id выкидываем исключение

=head2 SYNOPSIS

    Пример использования:
    my $tickets = get_tickets(['1', '3000000']); # кэширование включено
    my $ticket_for1 = $tickets->{'1'}
    my $tickets = get_tickets(['1', '3000000'], {scopes => '', cache_enabled => 0}); # кэширование выключено

=cut

sub get_tickets {
    my ($ids_array, $params) = @_;
    my $profile = Yandex::Trace::new_profile('tvm2:get_tickets');
    my ($cache_enabled, $scopes) = (1, '');
    if (exists $params->{cache_enabled}) {
        $cache_enabled = $params->{cache_enabled};
    }
    if (exists $params->{scopes}) {
        $scopes = $params->{scopes};
    }
    my @required_ids; # массив id приложений для которых нужно получить новые тикеты
    my %tickets; # будем складывать тикеты в этот hash
    if ($cache_enabled) {
        foreach my $id (@$ids_array) { # ищем тикеты в кэше. 
            ### and!
            if (exists($Cache{$APP_ID}{$id}{$scopes}) && time - $Cache{$APP_ID}{$id}{$scopes}[0] < $CACHE_TTL) {
                $tickets{$id} = $Cache{$APP_ID}{$id}{$scopes}[1];
            } else {
                push @required_ids, $id;
            }
        }
    } else {
        @required_ids = @$ids_array;
    }
    if (@required_ids == 0) { # новые тикеты не требуются
        return \%tickets;
    }
    my $new_tickets = _get_new_tvm2_tickets(\@required_ids, $scopes);
    %tickets = (%tickets, %$new_tickets); # собираем кэшированые и новые тикеты в один hash
    return \%tickets;
}


=head2 get_ticket($id, [$cache_enabled])

    Принимает id приложения для которого нужен тикет и hash с параметрами.
    Возможные параметры: 
        cache_enabled - если имеет значение false, то не ищем тикеты в кэше.
                        Если true или неопределен, то сначала ищем в кэше, если не нашли идем в сервис tvm2 по сети
        scopes - список прав
    Полученные тикеты кэшируются
    Возвращает запрошенный тикет.

=head2 SYNOPSIS

    Пример использования:
    my $ticket_for1 = get_ticket("2000086", {cache_enabled => 0}); # кэш выключен

=cut

sub get_ticket {
    my ($id, $params) = @_;
    return get_tickets([$id], $params)->{$id};
}


=head2 clear_tickets_cache()

    Очищает кэш.

=cut

sub clear_tickets_cache {
    %Cache = ();
}

1;
