package Yandex::URL;

=head1 NAME
    
    Yandex::URL

=head1 DESCRIPTION

    Функции для разнообразной обработки url'ов.
    Вся работа со ссылками происходится исключительно по

=cut

use strict;
use warnings;

use base qw/Exporter/;
our @EXPORT = qw /
    get_host
    get_second_level_domain
    get_top_level_domain
    get_num_level_domain
    strip_subdomain
    strip_protocol
    get_protocol
    clear_protocol
    get_domain_all_level_domains

    VALID_PROTOCOLS
/;

our @VALID_PROTOCOLS = qw/http https/;
=head2 get_host ($href)

	Принимает на вход ссылку и возвращает доменное имя.

=cut
sub get_host {
    my $href = shift;
    return '' if !defined $href;

    for ($href) {
        $href = strip_protocol($href); # strip "http://"
        s[[/?#;&:].*$][]; # strip path, port and parameters
    }

    return lc($href);
}

=head2 get_second_level_domain($href)

	Функция возвращает доменное имя, усеченное до второго уровня:
	(direct.yandex.ru => yandex.ru)

=cut

sub get_second_level_domain {
    my ($href) = shift;
    my $host = get_host($href);
    $host =~ s!.*\.([\w\d-]+\.[\w\d\-]+)!$1!;
    return $host;
}

=head2 strip_subdomain($href)

	Функция возвращает доменное имя без домена последнего уровня:
	(a.redirect.direct.yandex.ru => redirect.direct.yandex.ru)

=cut

sub strip_subdomain {
    my ($href) = shift;
    my $host = get_host($href);
    $host =~ s![\w\d\-]+\.?(.*)!$1!;
    return $host;
}

=head2 strip_protocol ($href)

	Принимает на вход ссылку и возвращает ее же, но без какого-либо протокола (http://, https://, ...)

=cut
sub strip_protocol {
    my $href = shift;
    return '' if !defined $href;
    $href =~ s[^[^/:]+://][];
    return $href;
}
=head2 get_protocol

    Принимает на вход ссылку и возвращает ее протокол (http://, https://, ...)

=cut 
sub get_protocol {
    my $href = shift;
    return '' if !defined $href;
    my ($protocol) = $href =~ m!(^[^/:]+)://!;
    return clear_protocol($protocol);
}

=head2 clear_protocol

    Принимает на вход протокол и подставляет значение по умолчанию, если протокол отсутствует. На валидность не проверяет.

=cut
sub clear_protocol {
    my $p = shift;
    $p ||= 'http://';
    my ($protocol) = $p =~ m!(^[^/:]+)(?:\://)?!;
    return $protocol.'://';

}

=head2 get_top_level_domain

	Возвращает имя домена верхнего уровня.

=cut
sub get_top_level_domain {
    return get_num_level_domain($_[0], 1);
}

=head2 get_num_level_domain(href, level)

    Возвращает заданный уровень домена из URL
    Например,
    
        get_num_level_domain("beta.direct.yandex.ru", 2) => 'yandex.ru'
        
        get_num_level_domain("beta.direct.yandex.ru", 3) => 'direct.yandex.ru'

=cut

sub get_num_level_domain {
    my ($href, $level_domain) = @_;
    
    $href || return '';
    
    my $domain = lc($href =~ /([^\/?:#]+)/ ? $1 : $href);
    my @parts =  reverse split(/\./, $domain);
    
    if (scalar @parts >= $level_domain) {
        @parts = splice @parts, 0, $level_domain;
    }
    
    return join ('.', reverse @parts);
}


sub get_domain_all_level_domains {
    my ($href, $min_level) = @_;
    $min_level = 1 unless $min_level;
    $href || return [];
    my @result;
    my $domain = lc($href =~ /([^\/?:#]+)/ ? $1 : $href);
    my @parts =  split(qr/\./, $domain);
    while (scalar(@parts) > $min_level) {
        push @result, join '.', @parts;
        shift @parts;
    }
    
    return \@result;
}

1;