package Yandex::Validate;

# $Id$

=head1 NAME

    Yandex::Validate
    разные простые проверки данных

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use Yandex::TimeCommon;
use String::Numeric qw/is_numeric is_float is_decimal is_integer is_int is_uint/;

use base qw/Exporter/;
our @EXPORT = qw/
                is_valid_int
                is_valid_id
                is_valid_time
                is_valid_date
                is_valid_ip
                is_valid_float
                is_valid_phone
                is_uint
                is_valid_base64
               /;

=head2 is_valid_int($val, $low_bound, $high_bound)
    
    Проверка скаляра на то, что он определён, содержит целое число, принадлежит указанному диапазону, 

=cut
my $valid_int_re = qr/^-?[0-9]+\z/;
sub is_valid_int {
    my ($val, $low_bound, $high_bound) = @_;
    return defined $val
        && $val =~ $valid_int_re 
        && (!defined $low_bound || $val >= $low_bound)
        && (!defined $high_bound || $val <= $high_bound);
}

=head2 is_valid_id($val)

    Проверка скаляра на то, что он содержит что-то, похожее на id - положительное целое число,
    помещающееся в 64 бита.

=cut

sub is_valid_id {
    my ($val) = @_;

    return unless is_valid_int( $val, 1 );

    # а здесь хитрый хак: если число не помещается в разрядность процессора, int $val превратит
    # его в стандартную форму (1.9e+19), так что eq вернёт ложное значение
    # - 2013-10-28 andy-ilyin
    return unless int $val eq $val;

    return 1;
}

=head2 is_valid_time($time_str)  

    Проверка строки на правильное описание времени формата HH:MM   

=cut

sub is_valid_time {
    my $time = shift;
    return $time =~ /^([0-9][0-9]):?([0-9][0-9])\z/ && $1 >= 0 && $1 <= 23 && $2 >= 0 && $2 <= 59;
}

=head2 is_valid_date($date_str)

    Проверка даты на валидность:
    1) формат dddd-mm-dd
    2) проблема 2038 года (дата должна быть меньше 2038-01-19)
    3) проверка "дата не в будущем" (опциональна)

=cut

my $valid_date_re = qr/^([0-9]{4})\-?([0-9]{2})\-?([0-9]{2})\z/;
sub is_valid_date {

    my $date = shift;

    my %OPTS = @_;

    unless ($date) {
        return 0;
    }

    my ($year, $month, $day) = $date =~ $valid_date_re;
    
    unless ( check_mysql_date($date) && $year >= 1970 && $year < 2038) {
        return 0;
    }

    # дата не в будущем
    if ($OPTS{no_future} && mysql2unix($date) > time()) {
        return 0;
    }

    return 1;
}

=head2 is_valid_ip($val)

    Проверка строки на соответствие записи IPv4

=cut
sub is_valid_ip {
    my $val = shift;
    my $token = qr/1?[0-9]{1,2}|2[0-4]\[0-9]|25[0-5]/;
    return $val =~ /^$token\.$token\.$token\.$token\z/ ? 1 : 0;
}

=head2 is_valid_float($val)

    Проверка числа с плавающей запятой

=cut

sub is_valid_float {
    my $val = shift;

    return is_float($val);
}

=head2 is_valid_phone

    Простая проверка на валидность введенного телефона:
        1) есть хотя бы 2 цифры
        2) телефон не содержит не печатымаемые символы

=cut

sub is_valid_phone {
    my $phone = shift || '';
    my @ciphers = $phone =~ /([0-9])/g;
    return $phone !~ /[^[:print:]]/ && scalar(@ciphers) >= 2 ? 1 : 0;
}

=head2 is_valid_base64

    Проверяет переданную строку на соответствие Base64 encoding

=cut

sub is_valid_base64 {
    my $str = shift;
    my $sanitized_str = join q{}, split /\n/, $str;
    return ($sanitized_str =~ m!^(?:[A-Za-z0-9+/]{4})*(?:[A-Za-z0-9+/]{2}==|[A-Za-z0-9+/]{3}=)?\z!);
}

1;
