package Yandex::Xiva;

# $Id$

=pod

=encoding utf-8

=head1 NAME

Yandex::Xiva - работа с пушами через Xiva

=head1 DESCRIPTION

    Подписка приложения на пуш-уведомления, отписка и отправка уведомлений.

=cut

use Direct::Modern;

use base qw/Exporter/;
our @EXPORT = qw/
    xiva_subscribe
    xiva_unsubscribe
    xiva_send_messages
/;

use Digest::SHA qw/sha224_hex/;
use File::Slurp qw/read_file/;
use JSON qw/to_json/;
use Yandex::HTTP qw/http_parallel_request/;
use Yandex::Log::Messages;
use Yandex::Trace;

our $XIVA_SUBSCRIPTION_TOKEN_FILE //= '/etc/direct-tokens/xiva_subscription';
our $XIVA_DISPATCH_TOKEN_FILE     //= '/etc/direct-tokens/xiva_dispatch';
our $SERVICE                      //= 'direct';
our $APP_NAME                     //= 'ru.yandex.direct';
our $FALLBACK_CLIENT_NAME         //= 'ru.yandex.direct.unknown';
our $XIVA_SERVICE_LOCATION        //= 'https://push.yandex.ru/v2/';
our $TIMEOUT                      //= 5;
our $RETRIES                      //= 2;
our $DEBUG_SYNC_SEND_MODE         //= 0; # при выставлении отправляет сообщения синхронным вызовом Xiva, журналируем ответ

=head2 _read_token

Прочитать токен из переданного пути и вернуть.

=cut

sub _read_token
{
    my $token = read_file(shift); # croaks on failure
    chomp $token;
    return $token;
}

=head2 _subscription_token

Получить токен подписки из секрета.

=cut

sub _subscription_token
{
    state $token //= _read_token($XIVA_SUBSCRIPTION_TOKEN_FILE);
    return $token;
}

=head2 _dispatch_token

Получить токен отправки из секрета.

=cut

sub _dispatch_token
{
    state $token //= _read_token($XIVA_DISPATCH_TOKEN_FILE);
    return $token;
}

=head2 _hash_token_for_log

Вернуть свёртку от токена, чтобы её можно было записать в лог.

=cut

sub _hash_token_for_log
{
    my $token = shift;
    return 'sha224:' . Digest::SHA::sha224_hex($token);
}

=head2 _xiva_call

Сделать вызов в нужную ручку.
Позиционные параметры:
    имя ручки,
    авторизационный токен для ручки в виде строки,
    хеш параметров строки адреса,
    хеш параметров тела запроса или строка.
Результат:
    HTTP-статус ответа или 0, если его нет.

=cut

sub _xiva_call
{
    my ($method, $auth_token, $args, $data) = @_;
    state $log //= Yandex::Log::Messages->new();
    my $base_url = $XIVA_SERVICE_LOCATION . $method;
    my $url = Yandex::HTTP::make_url($base_url, $args);

    my $headers = ($method eq 'send') ?
        {'Content-Type' => 'application/json; charset=utf-8'}
        : {'Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8'};
    if ($DEBUG_SYNC_SEND_MODE && $method eq 'send') {
        $headers->{'X-DeliveryMode'} = 'direct';
    }
    $headers->{'Authorization'} = 'Xiva ' . $auth_token;

    my $response = http_parallel_request(
        POST         => {1 => {url => $url, body => $data, headers => $headers}},
        timeout      => $TIMEOUT,
        num_attempts => $RETRIES
    )->{1};

    my $log_data = {%$args, base_url => $base_url};
    $log_data->{data} = $data;
    if ($method eq 'subscribe/app' && exists $log_data->{data}->{push_token}) {
        $log_data->{data}->{push_token_hash} = _hash_token_for_log($log_data->{data}->{push_token});
        delete $log_data->{data}->{push_token};
    }
    $log_data->{elapsed} = $response->{elapsed};
    $log_data->{response_headers} = $response->{headers};
    delete $log_data->{response_headers}->{URL};
    if ((($response->{headers}->{Status} // 0) != 200 || ($DEBUG_SYNC_SEND_MODE && $method eq 'send')) && $response->{content}) {
        $log_data->{response_content} = $response->{content};
    }
    $log->out($log_data);

    if ($response->{headers}->{Status}) {
        return $response->{headers}->{Status};
    };
    return 0;
}

=head2 xiva_subscribe

Подписать одно приложение на пуш-уведомления.
Именованные параметры:
    uid,
    client — строка с именем приложения и, возможно, версией (необязательный),
    uuid — id установки приложения,
    platform — apns или gcm,
    push_token,
    tag — выдуманный идентификатор устройства (необязательный, должен быть тогда,
        когда не указан device_id, не должен меняться без смены uuid или push_token),
    device_id — идентификатор устройства (необязательный).
Результат:
    статус HTTP или 0, если его нет.

=cut

sub xiva_subscribe
{
    my %O = @_;
    my $profile = Yandex::Trace::new_profile("xiva_subscribe");
    my $device  = $O{device_id} // $O{tag};
    return _xiva_call(
        'subscribe/app',
        _subscription_token(),
        {
            user     => $O{uid},
            client   => $O{client} // $FALLBACK_CLIENT_NAME,
            uuid     => $O{uuid},
            platform => $O{platform},
            service  => $SERVICE,
            app_name => $APP_NAME,
            device   => $device,
        },
        {
            push_token => $O{push_token},
        }
    );
}

=head2 xiva_unsubscribe

Отписать одно приложение от пуш-уведомлений.
Именованные параметры:
    uid,
    uuid — id установки приложения.
Результат:
    статус HTTP или 0, если его нет.

=cut

sub xiva_unsubscribe
{
    my %O = @_;
    my $profile = Yandex::Trace::new_profile("xiva_unsubscribe");
    return _xiva_call(
        'unsubscribe/app',
        _subscription_token(),
        {
            user    => $O{uid},
            uuid    => $O{uuid},
            service => $SERVICE
        },
        ''
    );
}

=head2 _xiva_send_one_message

Отправить одно уведомление.
Именованные параметры на входе: uid, message, payload, tag, device_id.
На выходе — статус HTTP.

=cut

sub _xiva_send_one_message
{
    my %O = @_;
    my $payload_copy = $O{payload} ? {%{$O{payload}}} : undef;
    my $event = ($payload_copy && $payload_copy->{T}) ? ("Type_" . $payload_copy->{T}) : "generic_event";
    my $device = $O{device_id} // $O{tag};
    my $formatted_body = {
        subscriptions => [{
            device => [$device]
        }],
        repack => {
            apns => {aps => {alert => $O{message}}}
        }
    };
    if ($payload_copy) {
        my @payload_keys = keys %$payload_copy;
        $payload_copy->{text} = $O{message};
        $formatted_body->{payload} = $payload_copy;
        $formatted_body->{repack}->{gcm}->{repack_payload} = [@payload_keys, 'text'];
        $formatted_body->{repack}->{apns}->{aps}->{'content-available'} = 1;
        if (exists $payload_copy->{category}) {
            $formatted_body->{repack}->{apns}->{aps}->{category} = $payload_copy->{category};
            @payload_keys = grep {$_ ne 'category'} @payload_keys;
        }
        $formatted_body->{repack}->{apns}->{repack_payload} = \@payload_keys;
    }

    return _xiva_call(
        'send',
        _dispatch_token(),
        {user => $O{uid}, event => $event},
        to_json($formatted_body)
    );
}

=head2 xiva_send_messages

Отправить несколько уведомлений
Параметры:
    ссылка на массив хешей
        uid,
        message — текст сообщения,
        payload — структура с содержимым payload,
        tag — выдуманный идентификатор устройства (необязательный, должен быть тогда,
            когда не указан device_id, не должен меняться без смены uuid или push_token),
        device_id — идентификатор устройства (необязательный).
Результат:
    ссылка на массив статусов HTTP или 0 для тех статусов, что не удалось получить.

=cut

sub xiva_send_messages
{
    my $args = shift;
    my $profile = Yandex::Trace::new_profile("xiva_send_messages", obj_num => scalar(@$args));
    return [map {_xiva_send_one_message(%{$_})} @$args];
}

1;
