package Yandex::SOAP::UTF8Serializer;

use strict;

use SOAP::Lite;
@Yandex::SOAP::UTF8Serializer::ISA = 'SOAP::Serializer';

use Encode qw//;

use Yandex::Trace;
use Data::Dumper;

=head2 new($enc)

    При вызове метода класса, создает новый сериализатор.
    В первом параметре принимает название кодировки. По умолчанию utf8.

    При вызове метода объекта, просто переводит вызов в родительский SOAP::Serializer,
    который позволяет через параметры менять свойства сериализатора.

=cut
sub new {
    my $self = shift;

    unless ( ref $self ) {
        my $class = $self;
        my $enc = shift;
        if ( !$enc ) {
            $enc = 'utf8';
        }

        $self = $class->SUPER::new( @_ );
        $self->{yandex_encoding} = $enc;
        return $self;
    }
    return $self->SUPER::new( @_ );
}

sub envelope {
    my ($self, @params) = @_;
    my $profile = Yandex::Trace::new_profile('utf8serializer:envelope');

    my $enveloped_data = $self->SUPER::envelope(@params);
    if ( !ref $enveloped_data && $self->{yandex_encoding} ne 'raw' ) {
        $enveloped_data = Encode::encode( $self->{yandex_encoding}, $enveloped_data );
    }
    return $enveloped_data;
}


sub as_base64 {
    my $self = shift;
    my($value, $name, $type, $attr) = @_;
    if ( !ref $value && $self->{yandex_encoding} ne 'raw' ) {
        $value = Encode::encode( $self->{yandex_encoding}, $value );
    }
    return $self->SUPER::as_base64( $value, $name, $type, $attr);
}


#sub encode_scalar {
#    my($self, $value, $name, $type, $attr) = @_;
#    if ( !ref $value && $self->{yandex_encoding} ne 'raw' ) {
#        $value = Encode::encode( $self->{yandex_encoding}, $value );
#    }
#
#    return $self->SUPER::encode_scalar( $value, $name, $type, $attr );
#}

# переопределение метода добавлено для коррекции неправильного типа элементов в пустом массиве
# Было
#   <SOAP-ENC:Array xsi:type="namesp2:ArrayOfCampaignInfo" SOAP-ENC:arrayType="xsd:ur-type[0]"/>
# Стало  
#   <SOAP-ENC:Array xsi:type="namesp2:ArrayOfCampaignInfo" SOAP-ENC:arrayType="namesp2:CampaignInfo[0]"/>
sub encode_array{
    my($self, $array, $name, $type, $attr) = @_;    
    my $result = $self->SUPER::encode_array( $array, $name, $type, $attr );
    if ($array && (scalar @$array) == 0){            
        if($result->[1]->{'xsi:type'} =~ /ArrayOf/){
            $result->[1]->{'SOAP-ENC:arrayType'} = $result->[1]->{'xsi:type'};
            $result->[1]->{'SOAP-ENC:arrayType'} =~ s/ArrayOf//; 
            $result->[1]->{'SOAP-ENC:arrayType'} .= '[0]';
        }
    }

    return $result;
}

sub disable_base64 {
    my ($self) = @_;
    my $typelookup = $self->{_typelookup};
    delete $typelookup->{base64};
    $self->typelookup($typelookup);
    return $self;
}

sub as_undef {
    $_[1] ? 'true' : 'false';
}

sub nilValue {
    'nil';
}

1;

package Yandex::SOAP::UTF8Deserializer;

use strict;

use SOAP::Lite;
@Yandex::SOAP::UTF8Deserializer::ISA = 'SOAP::Deserializer';

use Encode qw//;

use Yandex::Trace;

sub new {
    my $class = shift;
    my $enc = shift;
    if ( !$enc && ref $class ) { $enc = $class->{yandex_encoding}; }
    if ( !$enc ) { $enc = 'utf8'; }
    my $self = $class->SUPER::new( @_ );
    $self->{yandex_encoding} = $enc;

    # запрещаем использование external entity (дырка в безопасности)
    my $soap_parser = $self->{_parser};
    $soap_parser->{'_parser'} ||= $soap_parser->xmlparser;
    $soap_parser->{_parser}->setHandlers(
        ExternEnt => sub { shift; die "External entity (pointing to '$_[1]') is not allowed" },
        );

    bless $self;
}

sub deserialize {
    my ($self, @params) = @_;
    my $profile = Yandex::Trace::new_profile('utf8deserializer:deserialize');
    return $self->SUPER::deserialize(@params);    
}

sub decode_value {
    my $self = shift;
    my $res = $self->SUPER::decode_value( @_ );
    if ( !ref $res && !Encode::is_utf8($res)) {
        $res = Encode::decode( $self->{yandex_encoding}, $res );
    } elsif (ref $res && ref $res ne 'ARRAY' && ref $res ne 'HASH') {
        if (UNIVERSAL::isa($res, "ARRAY")) {
            $res = [@$res];
        } elsif (UNIVERSAL::isa($res, "HASH")) {
            $res = {%$res};
        }
    }
    return $res;
}

1;


package Yandex::SOAP::UTF8Deserializer::Masked;

use strict;
@Yandex::SOAP::UTF8Deserializer::Masked::ISA = 'Yandex::SOAP::UTF8Deserializer';

sub new {
    my $class = shift;
    my $self = $class->SUPER::new( @_ );
    bless $self;
}

sub fault {
    my $self = shift;
    my($code, $string, $detail, $actor) = @_;
          
    if( $code eq 'Server' ) {
        use Data::Dumper;
        print STDERR Dumper ['soap fault', \@_ ];
        $code = '500';
        $string = 'Ya Internal Server Error';
    }
    
    return $self->SUPER::fault( $code, $string, $detail, $actor );
}

1;


package Yandex::SOAP::UTF8Serializer::Masked;

use strict;
@Yandex::SOAP::UTF8Serializer::Masked::ISA = 'Yandex::SOAP::UTF8Serializer';

sub new {
    my $class = shift;
    my $self = $class->SUPER::new( @_ );
    bless $self;
}

sub fault {
    my $self = shift;
    my($code, $string, $detail, $actor) = @_;
    use Data::Dumper;

    if( $code eq 'Server' ) {
        print STDERR Dumper ['Server SOAP fault', \@_ ];
        $code = '500';
        $string = 'Yandex Internal Server Error';
        $detail = '';
        $actor = '';
    } elsif ( $code eq 'Client' ) {
        print STDERR Dumper ['Client SOAP fault', \@_ ];
        $code = '501';
        $string = 'Bad Soap Request';
        $detail = '';
        $actor = '';
    }
    return $self->SUPER::fault( $code, $string, $detail, $actor );
}

1;
