package Yandex::YacoTools;

# $Id$

=head1 NAME

Yandex::YacoTools

Работа с классификатором yacotools.yandex.ru

=head1 DESCRIPTION

=cut

use strict;
use base qw/Exporter/;

use LWP::UserAgent;
use HTTP::Request;
use URI::Escape qw/uri_escape_utf8/;
use Digest::MD5 qw/md5_hex/;
use YAML::Syck qw//;
use Data::Dumper;
use Encode;

use Yandex::HTTP;
use Yandex::Trace;

use utf8;

our @EXPORT_OK = qw/
    url_phrases
    ad_classify
    ads_classifier
/;

our $TIMEOUT ||= 60;
our $CATSHIFT = 9000000;
our $PHRASE_CLASSIFIER ||= 'http://catserver.yandex.ru:8886/direct';
our $AD_CLASSIFIER ||= 'http://yacotools.yandex.ru/yacotools/webapp/classifiers/classify_ad';
#our $PHRASE_CLASSIFIER = 'http://yacobot02t.yandex.ru:8886/direct';
#our $SUBQUERY_URL = "http://yacotools01t.yandex.ru/yacotools/webapp/classifiers/subquery";
our $SUBQUERY_URL ||= "http://yacotools.yandex.ru/yacotools/webapp/classifiers/subquery";

=head2 url_phrases

Классификация и нахождение фраз/привязок для веб-страницы по урлу.

пример: 

  url_phrases("http://www.rbc.ru/", {timeout => 10});

  => [ 
    { categories => [72], phrase => "курсы", rel => '0.002131' },
    { categories => [72,69], phrase => "валюта", rel => '0.98099' },
  ]

=cut
sub url_phrases {
    my ($url, $opt) = @_;

    my $profile = Yandex::Trace::new_profile('yacotools:url_phrases');

    # подготавливаем урл и опции
    $url =~ s/^\s+|\s+$//g;
    $url =~ s/(\s)/sprintf("%%%02X", ord($1))/ge;
    my $timeout = $opt->{timeout} || $TIMEOUT;

    # делаем запрос
    my $ua = new LWP::UserAgent(timeout => $timeout);
    my $req = HTTP::Request->new('POST' => $opt->{classifier} || $PHRASE_CLASSIFIER, undef, "$url\n\n");
    my $resp = $ua->request($req);
    if (!$resp->is_success) {
        die "YacoTools::url_phrases: Can't get phrases for $url: ".$resp->status_line;
    }

    # парсим ответ, проверяем на ошибки
    my $line = $resp->content;
    my ($md5, $http_status, $phrases_field) = split /\t/, $line;
    if (!$md5 || $md5 ne Digest::MD5::md5_hex($url)) {
        die "YacoTools::url_phrases: Incorrect md5 for $url: $line";
    } elsif (!$http_status || $http_status !~ /^\d+$/) {
        die "YacoTools::url_phrases: Incorrect http_status for $url: $line";
    } elsif ($http_status < 200 || $http_status >= 300) {
        die "YacoTools::url_phrases: Incorrect http_status from target site: $url -> $http_status";
    }

    # парсим набор фраз
    my @ret;
    for my $phrase_text (split ',', $phrases_field) {
        my ($rel, $categories, $phrase) = split ':', $phrase_text;
        push @ret, {
            phrase => Encode::decode_utf8($phrase),
            rel => $rel,
            categories => [map {$_ - 9000000} split / /, ($categories||'')]
        };
    }

    return \@ret;
}

=head2 ad_classify

Классификация набора фраз.

пример: 

  ad_classify(["юрист", "консультация"], {timeout => 1});
  
  => [ { id => 292, rough => 72, rel => 0.6850 } ]

=cut

sub ad_classify {
    my ($phrases, $opt) = @_;

    my $result = ads_classifier({"ad_1" => $phrases}, $opt)->{"ad_1"};
    return $result;
}

=head2 ads_classifier

Классификация набора групп фраз.

пример: 

  ads_classifier( {12345 => ["юрист", "консультация"], 98765 => ["кондицонер", "панасоник"]}, {timeout => 1});
  
  => {
          12345 => [{ id => 292, rough => 72, rel => 0.6850 }]
          , 98765 => [{ id => 194, rough => 13, rel => 1.6350 }]
     }

=cut

sub ads_classifier {
    my ($phrases, $opt) = @_;

    my $profile = Yandex::Trace::new_profile('yacotools:ads_classifier');

    # опции
    my $timeout = $opt->{timeout} || $TIMEOUT;

    my ($counter, @req, %dict) = (1);
    foreach my $label (keys %$phrases) {

        $dict{"ad_$counter"} = $label;

        push @req, map {"ad_$counter=".uri_escape_utf8($_)}
                    grep {defined $_ && $_ ne ""}
                        @{$phrases->{$label}};
        $counter++;
    }

    # делаем запрос
    my $ua = new LWP::UserAgent(timeout => $timeout);
    my $qs = join "&", @req;
    
    return {} if !$qs;

    my $req = HTTP::Request->new('POST' => $AD_CLASSIFIER, undef, $qs);
    # print STDERR $req->as_string;
    my $resp = $ua->request($req);
    if (!$resp->is_success) {
        die "YacoTools::ad_classify: Can't get phrases: ".$resp->status_line;
    }

    my ($cnt, %result) = (0);
    # print STDERR ">".$resp->content."<";
    # делим результат на отдельные сериализованные блоки
    foreach my $ad_res (split /\-{3}/, $resp->content) {
        # пропускаем первый пустой результат
        next unless $cnt++;

        # парсим ответ, проверяем на ошибки
        my $res = YAML::Syck::Load($ad_res);

        if (ref($res) ne 'ARRAY' || @$res != 2 || ref($res->[1]) ne 'ARRAY') {
            die "Incorrect classifier answer: ".$resp->content;
        }

        $result{ $dict{ $res->[0] } } = [
            map { 
                    {
                        id => ($_->{cat} - $CATSHIFT)
                        , rough => ($_->{cat_lev2} - $CATSHIFT)
                        , rel => $_->{closeness}
                    }
            } @{$res->[1]}
        ];
    }

    return \%result;
}


=head2 text_classify(hash, opt)

    Классификация текста объявления классификатором subquery.

    Формат данных: 
    	hash => {
    		'any key 1' => "Text for classification",
    		'any key 2' => "Text for classification",
    	}
	
	Возвращает:
    	{
    	    'any key 1' => [{cat => 123, weight => 33}, ...]
    	    'any key 2' => [{cat => 234, weight => 0.13}, ...]
    	}

=cut

sub text_classify
{
    my ($t, $opt) = @_;
    
    my $profile = Yandex::Trace::new_profile('yacotools:text_classify', obj_num => scalar(keys %$t));
	
	my ($counter, @request, %trans) = (1);
	foreach my $key (keys %$t) {
		$trans{"text_$counter"} = $key;
		push @request, "text_$counter=".$t->{$key};		
		++$counter;
	}
	
	$opt->{timeout} = defined $opt->{timeout} ? $opt->{timeout} : 10;
	
	my $post = Encode::encode_utf8(join '&', @request);
	my $txt = http_post($SUBQUERY_URL, $post, %$opt);
	my $result;
	if ($txt) {
		my $res = YAML::Syck::Load($txt);
        
		if ($res && ref($res) eq 'ARRAY') {
			foreach my $i (@$res) {
				my $key = $trans{$i->{key}};
				$result->{$key} = $i->{cats};
			}
		}
	}
	
	return $result;
}

1;
