package Yandex::YMaps;

=pod

    $Id$
    
    Модуль для работы с новым АПИ Яндекс.Карт

=cut

use strict;
use warnings;

use Data::Dumper;
use YAML::Syck;

use XML::LibXML;

use LWP::UserAgent;
use HTTP::Request;

use URI::Escape qw/uri_escape uri_escape_utf8/;

use Yandex::Trace;
use Yandex::HashUtils qw/hash_copy hash_merge/;

use Log::Any '$log';

#our $GEOCODE_URL = "http://geocode-maps.cloudkill.yandex.ru/1.x/?key=%s&geocode=%s";
our $GEOCODE_URL ||= "http://geocode.maps.yandex.net/1.x/?key=%s&geocode=%s";

our $GEOCODE_TIMEOUT ||= 10;



=head3 Конструктор

    Пример:
        my $geocode = Yandex::YMaps->new(KEY => $Settings::MKEYS{'direct.yandex.ru'});
        $geocode->search_address("Россия, Москва, ул.Самокатная, д.1, стр.21");

=cut

sub new 
{
    my $this = shift;
    
    my $class = ref($this) || $this;
    my $self = {@_};
    bless $self, $class;
    
    return $self;    
}

=head2 search_metro($x,$y, $options)

По координатам точки найти ближайшую станцию метро
$options - хеш опций для search_address

Возвращает объект от геокодера

=cut

sub search_metro
{
    my ($self, $x, $y, $opt) = @_;
    return undef unless $x and $y;
    my $res = $self->search_address("$x,$y", hash_merge {}, { kind => 'metro', results => 1}, $opt);
    if ($res and $res->{found} > 0) {
        return $res->{points}->[0];
    }
    return undef;
}

=head3 search_address(address_as_string)

    Делает запрос к геокодеру Яндекс.Карт для поиска указанного адреса
        и возвращает подготовленную для удобной обработки структуру:
        
    
    return  {
                found => # кол-во найденных адресов,
                request => # исходный запрос,
                points => [
                    {
                        country =>
                        city => 
                        street => 
                        house =>
                        build =>
                        
                        point => { # координата центральной точки, точки с найденным объектом
                            x =>
                            y =>
                        },
                        
                        bound => { # рекомендуемые границы области карты во время отображения
                            x1 =>
                            y1 =>
                            x2 =>
                            y2 =>  # где (x1,y1) - координаты нижнего угла 
                        }
                    }
                ]
            } || undef

=cut

sub search_address
{
    my $self = shift;
    my $address = shift || return;
    my $options = shift;
    
    my $data;
    my $url_params = hash_copy {}, $options, qw/kind results ll spn rspn skip lang/;
    
    if ($options->{safety}) {
        eval {
            my $tree = _parse_geo_xml($self->_get_geocoder_xml($address, $url_params));
            $data = _convert_maps_data($tree) || {};
        };
    } else {
        my $tree = _parse_geo_xml($self->_get_geocoder_xml($address, $url_params));
        $data = _convert_maps_data($tree) || {};
    }
    
    if (ref $data eq 'HASH' && $data->{found}) {
        return $data;
    }
    
    return;
}

=c Пример ответа геокодера (документация: http://api.yandex.ru/maps/geocoder/)
    
    GeocoderResponseMetaData
        request - исходный запрос
        found - кол-во найденных объектов
        suggest - если в запросе ошибка - то содержит исправленный вариант
        
    <featureMember>
        <GeoObject>
            <metaDataProperty>
                <GeocoderMetaData>
                    <kind>house</kind> - тип найденного объекта
                    <text>Россия, Москва, Энтузиастов шоссе, 9</text>
                    <precision>exact</precision> - точноть, м/б: exact(точное совпадение), number, near, street, other
                    <AddressDetails>
                        <Country>
                            <CountryName>Россия</CountryName>
                            <Locality>
                                <LocalityName>Москва</LocalityName>
                                <Thoroughfare>
                                    <ThoroughfareName>Энтузиастов шоссе</ThoroughfareName>
                                    <Premise>
                                        <PremiseNumber>9</PremiseNumber>
                                    </Premise>
                                </Thoroughfare>
                            </Locality>
                        </Country>
                    </AddressDetails>
                </GeocoderMetaData>
            </metaDataProperty>
        
            <boundedBy>
                <Envelope>
                    <lowerCorner>37.703586 55.747981</lowerCorner>
                    <upperCorner>37.720043 55.754172</upperCorner>
                </Envelope>
            </boundedBy>
        
            <Point>
                <pos>37.711815 55.751077</pos>
            </Point>
            
        </GeoObject>
    </featureMember>
=cut

=head3 _convert_maps_data(tree)

    Подготавливает более удобный для работы массив из XML дерева

    Формат возвращаемых данных:
      {
        found => int,
        points => [
            {
                kind =>
                precision =>
                text =>
                address => {
                    country => string
                    city => string
                    administrative_area => string
                    street => string
                    house =>
                    build =>
                },
                point => {
                    x, y => int
                },
                bound => {
                    x1, y1, x2, y2 => int
                }
                auto_precision => 'exact'
                auto_point => '37.727409,55.799947'
                auto_bounds => '37.723304,55.797635,37.731515,55.802260'
            }, .....
        ]
      }        

=cut

sub _convert_maps_data
{
    my $tree = shift || return;
    
    my $data;
    
    my $xp = new XML::LibXML::XPathContext;
    $xp->registerNs('gml', 'http://www.opengis.net/gml');
    $xp->registerNs('xal', 'urn:oasis:names:tc:ciq:xsdschema:xAL:2.0');
    $xp->registerNs('ygeo', 'http://maps.yandex.ru/geocoder/1.x');
    $xp->registerNs('ygeoint', 'http://maps.yandex.ru/geocoder/internal/1.x');
    $xp->registerNs('ymaps', 'http://maps.yandex.ru/ymaps/1.x');

    my @ymaps_nodes = $xp->findnodes('/ymaps:ymaps', $tree);
    my $ymaps = $ymaps_nodes[0];

    my @founds = $xp->findnodes('.//ygeo:found', $ymaps);
    $data->{found} = $founds[0]->textContent;

    if ($data->{found}) {
        
        my @fm_nodes = $xp->findnodes('.//gml:featureMember', $ymaps);
        foreach my $fm (@fm_nodes) {
            push @{$data->{points}}, Yandex::YMaps::xAL::get_address($xp, $fm);
        }        
    }
    
    return $data;
}

=head3 _convert_maps_data(address_string)

    Получает XML от геокодера Яндекс.Карт XML с результатом поиска указанного адреса

=cut
{
my %langs = (
    ru => 'ru-RU',
    ua => 'uk-UA',
    en => 'en-US', # американский английский
    tr => 'tr-TR'
);
sub _get_geocoder_xml
{
    my $self = shift;
    my $address = shift || return;
    my $url_options = shift || {};
    my $profile = Yandex::Trace::new_profile('ymaps:geocode');
    
    return if $address =~ /^\s*$/;
    
    my $url = sprintf ($GEOCODE_URL, $self->{KEY}, uri_escape_utf8($address));
    $url_options->{lang} = $langs{$url_options->{lang}} if $url_options->{lang} && exists $langs{$url_options->{lang}};
    if ($self->{ORIGIN}) {
        $url_options->{origin} = $self->{ORIGIN};
    }
    if (keys %$url_options) {
        $url .= '&';
        $url .= join '&', map { "$_=".uri_escape_utf8($url_options->{$_}) } keys %$url_options;
    }

    $log->info("Call $url");
    my $response = LWP::UserAgent->new(timeout => $GEOCODE_TIMEOUT)
                        ->request(HTTP::Request->new('GET', $url));
    
    if ($response->is_success) {
        #warn $response->content;
        return $response->content;
    } else {
        die "Geocode error: ".$response->status_line;
    }
    
    return;
}
}

=head3 _parse_geo_xml(xml_text)

    Преобразует текстовый XML в хэш

=cut

sub _parse_geo_xml
{
    my $text_xml = shift || return;
    
    my $el = XML::LibXML->new()->parse_string($text_xml);
    
    return $el;
}

1;

package Yandex::YMaps::xAL;

use XML::LibXML;

=pod

    Модуль для работы со структурами на языке 
        eXtensible Address Language (xAL)
        
    Спецификация: http://www.oasis-open.org/committees/ciq/ciq.html

    Язык используется для структурирования почтового адреса в ответе геокодера.
    
    В Я.Картах используется немного упрощенная версия, использующая лишь:
        Country, AdministrativeArea, Locality, Thoroughfare, Premise

=cut

=head3 get_address(xAL_data)

    Из структуры на xAL возвращает хэш с названием страны, города, улицы и номера дома/корпуса.

=cut

=head3

    Country::
        (AddressLine OR (CountryNameCode, CountryName, (AdministrativeArea OR Locality OR Thoroughfare)))
    
    AdministrativeArea::
        (AddressLines OR (AdministrativeAreaName, SubAdministrativeArea?, (Locality OR PostOffice OR PostalCode)))
    
    Locality::
        (AddressLines OR 
            (LocalityName, (PostBox OR LargeMailUser OR PostOffice), Thoroughfare, Premise, DependentLocality, PostalCode)) 
    
    Thoroughfare::
        (AddressLines OR (ThoroughfareName, ThoroughfarePreDirection, ThoroughfareLeadingType, ThoroughfareTrailingType, (ThoroughfareNumber OR ThoroughfareRange), ThoroughfareNumberType, ThoroughfareNumberSuffix, ThoroughfarePostDirection, DependentThoroughfare, ( DependentLocality OR Premise OR Firm OR PostalCode)))
    
    Premise::
        (AddressLines OR (PremiseName, (PremiseLocation OR PremiseNumber), PremiseNumberPrefix, PremiseNumberSuffix, BuildingName, (SubPremise OR Firm), PostalCode, Premise))

=cut

sub get_address
{
    my ($context, $featureMember) = @_;
    
    my $address = {};
    
    my @metas = $context->findnodes('.//gml:metaDataProperty', $featureMember);
    foreach my $meta (@metas) {
        $address->{geo_id} = get_text_value($context, './/ygeoint:geoid', $meta);

        foreach my $ygeo (qw/text kind precision/) {
            $address->{$ygeo} = get_text_value($context, ".//ygeo:$ygeo", $meta);
        }
        $address->{auto_precision} = $address->{precision};

        $address->{'country'} = get_text_value($context, './/xal:CountryName', $meta);
        $address->{'house'} = get_text_value($context, './/xal:PremiseNumber', $meta);

        $address->{'city'} = get_text_value($context, './/xal:LocalityName', $meta);
        $address->{'sub-city'} = get_text_value($context, './/xal:DependentLocality/xal:DependentLocalityName', $meta);

        $address->{'administrative-area'} = get_text_value($context, './/xal:AdministrativeAreaName', $meta);
        $address->{'sub-administrative-area'} = get_text_value($context, './/xal:SubAdministrativeArea/xal:SubAdministrativeAreaName', $meta);
        
        $address->{'street'} = get_text_value($context, './/xal:Thoroughfare/xal:ThoroughfareName', $meta);
        $address->{'sub-street'} = get_text_value($context, './/xal:DependentThoroughfare/xal:ThoroughfareName', $meta);
    }
    
    my @bound_nodes = $context->findnodes('.//gml:boundedBy', $featureMember);
    foreach my $bound (@bound_nodes) {
        ($address->{bound}{x1}, $address->{bound}{y1}) = split /\s+/, get_text_value($context, './/gml:lowerCorner', $bound);
        ($address->{bound}{x2}, $address->{bound}{y2}) = split /\s+/, get_text_value($context, './/gml:upperCorner', $bound);
    }
    $address->{auto_bounds} = join(",", $address->{bound}{x1}, $address->{bound}{y1}, $address->{bound}{x2}, $address->{bound}{y2});
    
    my @point_nodes = $context->findnodes('.//gml:Point', $featureMember);
    foreach my $point (@point_nodes) {
        ($address->{point}{x}, $address->{point}{y}) = split /\s+/, get_text_value($context, './/gml:pos', $point);
    }
    $address->{auto_point} = join(",", $address->{point}{x}, $address->{point}{y});
    
    return $address;
}

sub get_text_value
{
    my ($context, $xpath, $node) = @_;
    
    my @nodes = $context->findnodes($xpath, $node);
    
    return @nodes ? $nodes[0]->textContent : undef; 
}
