package Yandex::YT::Streaming;

use strict;
use warnings;
use Carp qw/croak/;

use Yandex::YT::Format::dsv;
use Yandex::YT::Format::json;

our $DEFAULT_FORMAT ||= 'dsv';
our $DEFAULT_FORMAT_FLAGS ||= {
    json => 'encode_utf8=false',
};

sub new {
    my ($class, %O) = @_;
    my $self = {};
    if (my $reduceby = delete $O{reduceby}) {
        $self->{reduceby} = $reduceby;
        if (ref $reduceby eq 'ARRAY') {
            $self->{reduceby_sub} = sub {
                my $rec = shift;
                return join '|', map {defined $rec->{$_} ? $rec->{$_} : ''} @$reduceby;
            };
        } else {
            croak 'Incorrect reduceby argument';
        }
    }

    $self->{in_fh} = delete $O{in_fh} || \*STDIN;
    $self->{out_fh} = delete $O{out_fh} || \*STDOUT;

    $self->{format} = delete($O{format}) || $DEFAULT_FORMAT;
    $self->{loads_sub} = \&{"Yandex::YT::Format::$self->{format}::loads"};
    $self->{dumps_sub} = \&{"Yandex::YT::Format::$self->{format}::dumps"};

    croak "Excess params to Yandex::YT::Streaming->new: ".join(', ', keys %O) if %O;
    
    bless $self;
}

sub get {
    my ($self) = @_;
    
    return delete $self->{reduce_rec} if $self->{reduce_rec};

    my $rec;
    while(1) {
        $self->{_line} = $self->{in_fh}->getline();
        return undef if !defined $self->{_line};
        chomp $self->{_line};
        $rec = $self->{loads_sub}->($self->{_line});
        if ($self->{format} eq 'dsv' && exists $rec->{'@table_index'}) {
            $self->{table_index} = delete $rec->{'@table_index'};
        }
        if (exists $rec->{'$value'} && !defined $rec->{'$value'}) {
            $self->{_attrs} = $rec->{'$attributes'};
            $self->{table_index} = $self->{_attrs}->{table_index};
            next;
        } else {
            last;
        }
    }
    
    return $rec if !defined $self->{reduceby_sub};
    
    my $reduce_key = $self->{reduceby_sub}->($rec);
    if (defined $self->{reduce_key} && $self->{reduce_key} eq $reduce_key) {
        return $rec;
    }

    $self->{reduce_key} = $reduce_key;
    $self->{reduce_rec} = $rec;
    $self->{reduce_group} = {map {$_ => $rec->{$_}} @{$self->{reduceby}}};
    
    return undef;
}

sub get_all {
    my ($self) = @_;
    my @ret;
    while(my $rec = $self->get) {
        push @ret, $rec;
    }
    return @ret;
}

sub get_group {
    my ($self, $key) = @_;
    
    if (!$self->{reduce_group}) {
        $self->get();
    }
    
    return delete $self->{reduce_group};
}

=head2 $s->pass_group()

    Пропустить все записи группы, не обрабатывая
    Следующий get() вернёт запись из новой группы

=cut
sub pass_group {
    my ($self) = @_;
    while(my $rec = $self->get) {
        1;
    }
}

sub yield {
    my ($self, $rec, $table_number) = @_;
    
    my $out_fh;
    if (defined $table_number && $table_number > 0) {
        $self->{extra_out_fhs} //= {};
        if (!exists $self->{extra_out_fhs}->{$table_number}) {
            my $fd = 3 * $table_number + 1;
            open(my $fh, ">&$fd") || croak "Can't open stream $fd: $!";
            $self->{extra_out_fhs}->{$table_number} = $fh;
        }
        $out_fh = $self->{extra_out_fhs}->{$table_number};
    } else {
        $out_fh = $self->{out_fh};
    }
    $out_fh->print($self->{dumps_sub}->($rec)."\n");
}

sub finish {
    my ($self) = @_;
    
    return unless $self->{extra_out_fhs};

    for my $table_number (keys %{ $self->{extra_out_fhs} }) {
        my $fh = $self->{extra_out_fhs}->{$table_number};
        my $fd = 3 * $table_number + 1;
        close($fh) || croak "Can't close stream $fd: $!";
    }
}

sub table_index {
    my $self = shift;
    return $self->{table_index};
}

1;
