package Yandex::YT::Table;

=head1 NAME

    Yandex::YT::Table - данные о таблице из YT

=head1 SYNOPSIS

    
    my $table = Yandex::YT::Table->new('//home/direct/db/bids', binmode => ':utf8');

    my $attr_value = $table->get_attribute('modification_time');
    my $reader = $table->reader();  - объект типа Yandex::YT::TableReader
    $reader = $table->reader([qw/id price/], "[:#10]");  - объект типа Yandex::YT::TableReader, с выборкой полей и диапазона

=cut

use strict;
use warnings;

use Carp;
use JSON;

use Yandex::Shell;
use Yandex::YT::Table::Schema;
use Yandex::YT::TableReader;

sub new {
    my ($cls, $table_name, %opt) = @_;

    $cls = ref($cls) || $cls;

    $table_name =~ s/\/$//;
    my $config = delete $opt{config};
    croak "Unsupported options: ".join(',', keys %opt) if %opt;
    bless {table_name => $table_name, opt => \%opt, config => $config}, $cls;
}

=head2 get_attribute

    Возвращает атрибуты таблицы
    На входе:
        attribute - название атрибута
    На выходе:
        значение атрибута, строка.

=cut
sub get_attribute {
    my ($self, $attribute) = @_;
    return $self->get_attributes->{$attribute};
}

=head2 get_attributes

    Возвращает все атрибуты таблицы
    На выходе:
        ссылка на хэш

=cut
sub get_attributes {
    my ($self) = @_;
    my $result = $self->_run_yt_cmd('get', '--format=json', "$self->{table_name}/\@");
    return from_json($result);
}

=head2 set_attribute

    Устанавливает атрибут таблицы
    На входе:
        attribute - название атрибута
        value - значение атрибута

=cut
sub set_attribute {
    my ($self, $attribute, $value) = @_;
    $self->_run_yt_cmd('set', "$self->{table_name}/\@$attribute", qq!"$value"!);
}


=head2 reader

    На входе:
        fields - список полей, которые требуется выбрать (array_href)
        interval - диапазон выборки (строка в синстаксисе YT)
        opt - опции, которые будут переданы в конструктор Yandex::YT::TableReader
    На выходе:
        указатель на Yandex::YT::TableReader

=cut
sub reader {
    my ($self, $fields, $interval, %opt) = @_;
    # TableReader просит параметры такого вида: '//home/direct/db/bids{id,price}[:#10]'
    # где //home/direct/db/bids - было указано в конструкторе Yandex::YT::Table;
    my @params = ($self->{table_name});
    push @params, sprintf('{%s}', join ',', @$fields) if defined $fields && @$fields;
    push @params, $interval if defined $interval;

    return Yandex::YT::TableReader->new(join('', @params), config => $self->{config}, %opt);
}


=head2 exists

    Проверяет, существует ли нода
    На выходе:
        0/1

=cut
sub exists {
    my ($self) = @_;
    return $self->_run_yt_cmd('exists', $self->{table_name}) =~ /^true/i ? 1 : 0;
}


=head2 create([node_type],[attrs_str])

    Cоздаёт ноду указанного типа, по-умолчанию - table
    Нода создаётся с рекурсивным созданием путей
    Уже существующая нода игнорируется

=cut
sub create {
    my ($self, $node_type, $attrs_str) = @_;
    $node_type //= 'table';
    $self->_run_yt_cmd('create', '-r', '-i', $node_type, $self->{table_name}, 
                       $attrs_str ? ( '--attributes', $attrs_str ) : ());
}

=head2 create_table_with_schema($schema, $use_new_chunk_format)

    Создает таблицу со строгой схемой хранения.
    https://wiki.yandex-team.ru/yt/userdoc/staticschema/

    Формат $schema смотри в Yandex::YT::Table::Schema
    
    Можно передать последним аргументом $use_new_chunk_format = 0
    для хранения данных в старом (построчном) формате.
    по умолчанию используется новый формат - поколоночный.
    
    Нода создаётся с рекурсивным созданием путей.
    Уже существующая нода игнорируется.

=cut

sub create_table_with_schema {
    my ($self, $schema, $use_new_chunk_format) = @_;
    $use_new_chunk_format //= 1;

    if (my $error = Yandex::YT::Table::Schema::validate($schema)) {
        die $error;
    }

    my $attrs_str = '{ ';
    $attrs_str .= 'optimize_for = ' . ($use_new_chunk_format ? 'scan' : 'lookup') . '; ';
    $attrs_str .= 'schema = ' . Yandex::YT::Table::Schema::get_attribute_string($schema);
    $attrs_str .= ' }';

    $self->_run_yt_cmd('create', '-r', '-i', 'table', $self->{table_name}, '--attributes', $attrs_str);
}

=head2 list()

    Возвращает список нод находящихся внутри директории, которая представлена текущим объектом.
    Список содержит объекты типа Yandex::YT::Table

=cut

sub list {
    my ($self, %opt) = @_;
    my $list = $self->_run_yt_cmd( "list" => $self->{table_name} );
    return map {chomp; $self->new( $self->{table_name}.'/'.$_, config => $self->{config}) } (split/\n/, $list);
}

=head2 node_name()

    Возвращает имя ноды представленной текущим объектом

=cut

sub node_name
{
    my ($self) = @_;
    return $self->{table_name};
}

=head2 move(new_path)

Перемещает таблицу в Ыть

=cut

sub move {
    my ($self, $new_path) = @_;
    $self->_run_yt_cmd('move', $self->node_name, $new_path->node_name);
}

=head3 _run_yt_cmd(...)

Запускает yt через yash_qx. Если у объекта указан конфиг, добавляет его параметром в командную строку

=cut

sub _run_yt_cmd {
    my ($self, @rest) = @_;

    my @global_params;
    if ($self->{config}) {
        push @global_params, ('--config' => $self->{config});
    }

    return yash_qx('yt', @global_params, @rest);
}


1;
