package Yandex::YT::Table::Schema;

=head1 NAME

    Yandex::YT::Table::Schema - работа с информацией о схеме таблиц

=head1 SYNOPSIS

    if (my $error = Yandex::YT::Table::Schema::validate($schema)) {
        die $error;
    }
    my $schema_str = Yandex::YT::Table::Schema::get_attribute_string($schema);

    yt create table path --attributes "{schema = $schema_str}"

=head1 DESCRIPTION

=head2 Формат $schema

    [
        {
            name => "column1",
            type => "any",
            group => "group1",
        }, {
            name => "column2",
            type => "string",
            group => "group1",
        }, ...
    ]

    Список значений для форматов - можно подсмотреть ниже в %VALID_COLUMN_TYPES
    Указание группы - опционально.
    Указание признака сортировки по ключу - не поддерживается.

=cut

use warnings;
use strict;

=head2 %VALID_COLUMN_TYPES

    Допустимые типы для колонок
    # https://wiki.yandex-team.ru/yt/userdoc/tables/#sxema

=cut

my %VALID_COLUMN_TYPES = map {$_ => undef} qw/
    int64
    uint64
    double
    boolean
    string
    any
/;

=head2 validate($schema)

    Валидирует описание схемы для таблицы.
    Если все ок - возвращает undef, при ошибках - строку с описанием первой ошибки.

=cut

sub validate {
    my ($schema) = @_;

    if (!$schema || ref $schema ne 'ARRAY') {
        return 'invalid schema format, should be arrayref';
    }

    my $i = 0;
    for my $column (@$schema) {
        if (!$column || ref $column ne 'HASH') {
            return "invalid specification for column #$i, should be hashref";
        }

        if (!$column->{name} || $column->{name} !~ m/^[a-z0-9_-]+$/i) {
            return "name for column #$i is not specified or has invalid format";
        }

        if (!$column->{type} || !exists $VALID_COLUMN_TYPES{ $column->{type} }) {
            return "type for column #$i ($column->{name}) is invalid or not specified, should be one of: " . join(", ", sort keys %VALID_COLUMN_TYPES);
        }

        if ($column->{group} && $column->{group} !~ m/^[a-z0-9_-]+$/i) {
            return "group for column #$i ($column->{name}) has invalid format";
        }

        $i++;
    }

    if ($i == 0) {
        return "no columns specified";
    }
    
    return undef;
}

=head2 get_attribute_string($schema)

    По объекту $schema получить строку - значение аттрибута schema

=cut

sub get_attribute_string {
    my ($schema) = @_;

    my $schema_str = '<strict=%true>[';
    for my $column (@$schema) {
        $schema_str .= '{ ';
        $schema_str .= "name = $column->{name}; ";
        $schema_str .= "type = $column->{type}; ";
        if ($column->{group}) {
            $schema_str .= "group = $column->{group}; ";
        }
        $schema_str .= ' }; ';
    }
    $schema_str .= '];';

    return $schema_str;
}
