package main

import (
	"fmt"
	"os/exec"
	"syscall"
	"os"
        "strings"
        "errors"
	"bytes"
)

func print_result(command string) {
	var out string
	var err error
	switch command {
	case "list":
	    out, err = list_snapshots()
	case "pool_list":
            out, err = list_pools()
	case "pool_status":
	    out, err = status_pools()
        }
	if err != nil {
		fmt.Printf("%s\n", err)
		os.Exit(1)
	} else {
		fmt.Printf("%s\n", out)
		os.Exit(0)
	}
}

func exec_command(command string) (result string, err error) {
        var buf bytes.Buffer
	var out []byte
	lcmd := strings.Split(command, " ")
	cmd := exec.Command(lcmd[0], lcmd[1:]...)
	cmd.SysProcAttr = &syscall.SysProcAttr{}
	cmd.SysProcAttr.Credential = &syscall.Credential{Uid: 0, Gid: 0}
	out, err = cmd.Output()
	buf.Write(out)
	return buf.String(), err
}

func exec_combined(command string) (err error) {
	var msg []byte
	lcmd := strings.Split(command, " ")
	cmd := exec.Command(lcmd[0], lcmd[1:]...)
	cmd.SysProcAttr = &syscall.SysProcAttr{}
	cmd.SysProcAttr.Credential = &syscall.Credential{Uid: 0, Gid: 0}
	if msg, err = cmd.CombinedOutput(); err != nil {
		err = errors.New(string(bytes.TrimSpace(msg)))
	}
	return err
}

func list_snapshots() (result string, err error) {
	return exec_command("/sbin/zfs list -t snap")
}

func list_mounts() (result string, err error) {
	return exec_command("/bin/mount -t zfs")
}

func list_pools() (result string, err error) {
	return exec_command("/sbin/zpool list")
}

func status_pools() (result string, err error) {
	return exec_command("/sbin/zpool status")
}

func mount_snapshot(name string) (err error) {
        var out string
        if out, err = list_snapshots(); err != nil {
            return err
        }
        if ok := strings.Contains(out, name + " "); !ok {
            return errors.New(fmt.Sprintf("[%s] не найден снапшот для монтирования", name))
        }
        if out, err = list_mounts(); err != nil {
            return err
	}
	if ok := strings.Contains(out, name + " "); ok {
	    return errors.New(fmt.Sprintf("[%s] снапшот уже смонтирован", name))
        }
	cmd := fmt.Sprintf("/bin/mount -t zfs %s /mnt", name)
	return exec_combined(cmd)
}

func unount_snapshot(name string) (err error) {
	var out string
	var msg []byte
	if out, err = list_mounts(); err != nil {
	    return err
	}
	if ok := strings.Contains(out, name + " "); !ok {
	    return errors.New(fmt.Sprintf("[%s] в указанной точке отмонтирования нет снапшота", name))
	}
        cmd := exec.Command("/bin/umount", name)
	cmd.SysProcAttr = &syscall.SysProcAttr{}
	cmd.SysProcAttr.Credential = &syscall.Credential{Uid: 0, Gid: 0}
	if msg, err = cmd.CombinedOutput(); err != nil {
	    err = errors.New(string(bytes.TrimSpace(msg)))
	}
	return err
}

func main() {
       var USAGE = `Утилита для просмотра списка ZFS снапшотов, монтирования и размонтирования.
       zfsctl list - покажет список доступных снапшотов
       zfsctl mount <имя снапшота> - смонтирует указанный снапшот в /mnt
       zfsctl umount <имя снапшота> - размонтирует указанный снапшот из /mnt
       zfsctl pool_list - список пулов zfs
       zfsctl pool_status - статус zfs пулов 
       Удачи!! 
       
       P.S. Вам может показаться, что приложение написано на GoLang, но это не так! ^_^`
       var errors []error
       if len(os.Args) == 1 {
	       fmt.Println(USAGE)
	       os.Exit(0)
       }
       switch cmd := os.Args[1]; cmd {
       case "list":
           print_result(cmd)
       case "pool_list":
	   print_result(cmd)
       case "pool_status":
           print_result(cmd)
       case "mount":
	   if len(os.Args) < 3 {
		   fmt.Println("не указано, что монтировать")
		   os.Exit(1)
	   }
	   for _, point := range os.Args[2:] {
		   if err := mount_snapshot(point); err != nil {
			   errors = append(errors, err)
		   }
	   }
       case "umount":
	   if len(os.Args) < 3 {
	       fmt.Println("не указано, что отмонтировать")
	       os.Exit(1)
	   }
           for _, point := range os.Args[2:] {
		   if err := unount_snapshot(point); err != nil {
			   errors = append(errors, err)
		   }
	   }
       default:
	   fmt.Println(USAGE)
	   os.Exit(0)
       }
       if len(errors) > 0 {
	       for _, err := range errors {
	            fmt.Println(err)
	       }
	       os.Exit(1)
       }
}
