#!/usr/bin/python
# -*- encoding: utf-8; -*-

import os.path
import os
import time
import optparse
import yaml
import logging

import threading
from subprocess import Popen, PIPE

import zookeeper


DEFAULT_RECONNECT_PERIOD = 3600
DEFAULT_ITERATION_PERIOD = 30


def write_file(filename, cont=''):
    open(filename + '.tmp', 'w').write(cont)
    os.rename(filename + '.tmp', filename)


def read_file(filename):
    return open(filename, 'r').read()


def zk_sync_init(host, fn=None, recv_timeout=10000, clientid=None):
    """
    Синхронный вариант коннекта к zookeeper - возврат происходит либо
    после успешного коннекта, либо по истечении таймаута
    Параметры аналогичны zookeeper.init
    """
    cv = threading.Condition()
    def watcher(handle, type, state, path):
        if fn:
            fn(handle, type, state, path)
        cv.acquire()
        cv.notifyAll()
        cv.release()
    cv.acquire()
    zkh = zookeeper.init(host, watcher, recv_timeout, clientid if clientid else (-1, ""))
    cv.wait(3.0 * recv_timeout / 1000 if recv_timeout else 60)
    cv.release()
    return zkh


class FilesDeliveryService(object):
    def __init__(self, config_file):
        self.config_file = config_file
        self.config_file_stat = None
        self.conf = None
        self.zkh = None


    def read_config(self):
        self.config_file_stat = os.stat(self.config_file)
        self.conf = yaml.safe_load(open(self.config_file))
        self.alive_file = self.conf.get('alive_file', None)

    def is_config_changed(self):
        config_file_stat = os.stat(self.config_file)
        return config_file_stat.st_ino != self.config_file_stat.st_ino \
            or config_file_stat.st_size != self.config_file_stat.st_size \
            or config_file_stat.st_mtime != self.config_file_stat.st_mtime

    def run(self):
        # при недоступном zookeeper-e крутимся в цикле
        self.one_iteration()
        while(True):
            time.sleep(self.conf.get('iteration_period', DEFAULT_ITERATION_PERIOD))
            if zookeeper.state(self.zkh) != zookeeper.CONNECTED_STATE:
                logging.debug("Reconnect to zookeeper (disconnected)")
                self.one_iteration()
            elif time.time() - self.iteration_start > self.conf.get("reconnect_period", DEFAULT_RECONNECT_PERIOD):
                logging.debug("Reconnect to zookeeper (by time)")
                self.one_iteration()
            if self.alive_file:
                write_file(self.alive_file)
            if self.is_config_changed():
                logging.debug("config changed")
                zookeeper.close(self.zkh)
                self.zkh = None
                self.one_iteration()

    def one_iteration(self):
        logging.debug("one_iteration start")
        self.iteration_start = time.time()
        self.read_config()
        self.configure_logging()
        if self.zkh:
            zookeeper.close(self.zkh)
            self.zkh = None
        logging.debug('new iteration')
        zookeeper.set_log_stream(self.logfh)
        logging.info('connect to %s' % self.conf['servers'])
        self.zkh = zk_sync_init(self.conf['servers'], None, 10000)
        if 'auth' in self.conf:
            zookeeper.add_auth(zkh, 'digest', self.conf['auth'], None)
        for finfo in self.conf['files']:
            self.get_and_watch_file(finfo)


    def configure_logging(self):
        oldmask = os.umask(0)
        self.logfh = open(self.conf['log'], 'a', 0666)
        os.umask(oldmask)

        root = logging.getLogger()
        if root.handlers:
            for handler in root.handlers:
                root.removeHandler(handler)
        logging.basicConfig(stream=self.logfh,
                            level=logging.INFO,
                            format='%(asctime)s ['+str(os.getpid())+'] %(levelname)s %(message)s'
                            )


    def get_and_watch_file(self, finfo):
        def watcher(handle, state, type, zk_path):
            self.get_and_watch_file(finfo)
        file_cont, meta = zookeeper.get(self.zkh, finfo['zk_path'], watcher)
        f = finfo['file']
        if not os.path.exists(f) or file_cont != read_file(f):
            logging.debug("file %s changed" % f)
            if 'hooks_status_file' in finfo and finfo['hooks_status_file']:
                write_file(finfo['hooks_status_file'], "running")
            logging.info("write to file '%s' data '%s'" % (f, file_cont))
            write_file(f, file_cont)
            self.process_hooks(finfo, meta)
        else:
            logging.debug("File %s is not changed" % f)


    def process_hooks(self, finfo, meta):
        if 'hooks_dir' not in finfo or not finfo['hooks_dir'] or not os.path.isdir(finfo['hooks_dir']):
            if 'hooks_status_file' in finfo and finfo['hooks_status_file']:
                os.unlink(finfo['hooks_status_file'])
            return
        hook_failed = False
        for hook_file in os.listdir(finfo['hooks_dir']):
            hook_file_abs = os.path.join(finfo['hooks_dir'], hook_file)
            if ".dpkg" in hook_file:
                logging.debug("skip %s: filename contains .dpkg" % hook_file_abs)
                continue
            elif not os.path.isfile(hook_file_abs):
                logging.debug("skip %s: not a file" % hook_file_abs)
                continue
            elif not os.access(hook_file_abs, os.X_OK):
                logging.debug("skip %s: not executable file" % hook_file_abs)
                continue
            logging.info("Run hook %s" % hook_file_abs)
            # в хуки передаём параметрами командной строки: путь к файлу с данными, версию из zookeeper
            # контракт может измениться в будущем
            p = Popen([hook_file_abs, finfo['file'], str(meta['version'])], stdout=PIPE, stderr=PIPE)
            out, err = p.communicate()
            if p.returncode:
                # fail
                logging.error("Hook %s return exit status %d, stdout: %s, stderr: %s" % (hook_file_abs, p.returncode, out, err))
                hook_failed = True
            else:
                logging.info("Hook %s return exit status %d, stdout: %s, stderr: %s" % (hook_file_abs, p.returncode, out, err))
        if 'hooks_status_file' in finfo:
            if hook_failed:
                write_file(finfo['hooks_status_file'], "failed")
            else:
                os.unlink(finfo['hooks_status_file'])


def main():
    # парсим опции
    usage = "usage: zk-delivery -c config-filename"
    parser = optparse.OptionParser(usage=usage)
    parser.add_option("-c", "--config",
                  action="store", dest="config",
                  help="Config filename")
    (options, args) = parser.parse_args()
    if args:
        parser.error("Extra arguments defined: ")
    elif options.config == None:
        parser.error("Config is not defined")
    elif not os.path.exists(options.config):
        parser.error("Config file does not exists");

    FilesDeliveryService(options.config).run()


if __name__ == '__main__':
    main()
