#!/usr/bin/python
# -*- encoding: utf-8; -*-

import os.path
import base64
import hashlib
import os
import sys
import re
import logging
import time
import optparse
import yaml
import simplejson
import zookeeper

from zkdelivery import zk_sync_init

ZOO_OPEN_ACL_UNSAFE = [{"perms":0x1f, "scheme":"world", "id" :"anyone"}]
ZOO_OPEN_ACL_LOCAL = [{"perms":0x1f, "scheme":"ip", "id" :"127.0.0.1"}, {"perms":0x1, "scheme":"world", "id" :"anyone"}]

def parse_options():
    # парсим опции
    usage = "usage: zk-delivery-set -c config-filename zk-node-name local-filename\n" \
          + "       zk-delivery-set -c config-filename zk-node-name --data 'data'\n" \
          + "       zk-delivery-set -c config-filename zk-node-name --json db_config.CHILDS.ppc.host [www,www2]\n" \
          + "       zk-delivery-set -c config-filename zk-node-name --yaml db_config.CHILDS.ppc.host [www,www2]\n" \
          + "       zk-delivery-set -c config-filename zk-node-name --data 'data' --create"
    parser = optparse.OptionParser(usage=usage)
    parser.add_option("-c", "--config",
                  action="store", dest="config",
                  help="Config filename")
    parser.add_option("--log",
                  action="store", dest="log",
                  help="Override logfile name from config")
    parser.add_option("--log-level",
                  action="store", dest="log_level", default="info",
                  help="Override logfile name from config")
    parser.add_option("--create",
                  action="store_true", dest="create",
                  help="Create node, if not exists and set")
    parser.add_option("--data",
                  action="store", dest="data",
                  help="Data to set")
    parser.add_option("--yaml",
                  action="append", nargs=2, dest="yaml", metavar="YAMLKEY VALUE",
                  help="Special method to set value in yaml file")
    parser.add_option("--json",
                  action="append", nargs=2, dest="json", metavar="JSONKEY VALUE",
                  help="Special method to set value in json file (understand yaml and json values)")
    options, args = parser.parse_args()
    return parser, options, args

def check_options(parser, options, args):
    if options.config == None:
        parser.error("Config is not defined")
    elif not os.path.exists(options.config):
        parser.error("Config file does not exists");

def set_logging(log, log_level):
    error = None
    oldmask = os.umask(0)
    try:
        logfh = open(log, 'a', 0666)
    except IOError as e:
        logfh = sys.stdout
        error = e
    finally:
        os.umask(oldmask)

    log_level = getattr(logging, log_level.upper())
    logging.basicConfig(stream=logfh,
                        level=log_level,
                        format='%(asctime)s ['+str(os.getpid())+'] %(levelname)s %(message)s'
                        )
    zookeeper.set_log_stream(logfh)
    zookeeper.set_debug_level(zookeeper.LOG_LEVEL_ERROR)
    if error is not None:
        logging.error(error, exc_info=True)
        logging.warn("fallback logging to stdout")


def get_acl(conf):
    def cred_digest(auth):
        return auth.split(':')[0] + ':' + base64.b64encode(hashlib.sha1(auth).digest())
    if 'acl' in conf:
        acl = []
        for key, perms in dict(read=0x1, write=0x1f).items():
            for item in conf['acl'].get(key, []):
                if item == 'anyone':
                    scheme = 'world'
                elif re.match(r'^\d+\.\d+\.\d+\.\d+(/\d+)?$', item):
                    scheme = 'ip'
                elif re.match(r'^\w+:.+$', item):
                    scheme = 'digest'
                    item = cred_digest(item)
                else:
                    raise Exception('incorrect acl: %s' % item)
                acl.append({"perms":perms, "scheme":scheme, "id":item})
    elif conf.get('local_server', None):
        acl = ZOO_OPEN_ACL_LOCAL
    else:
        acl = ZOO_OPEN_ACL_UNSAFE

    if 'auth' in conf:
        digest = conf['auth'].split(':')[0] + ':' + base64.b64encode(hashlib.sha1(conf['auth']).digest())
        acl.append({"perms":0x1f, "scheme":'digest', "id":cred_digest(conf['auth'])})

    return acl


def main():
    # парсим опции
    parser, options, args = parse_options()
    check_options(parser, options, args)

    conf = yaml.safe_load(open(options.config))
    set_logging(options.log or conf['log'], options.log_level)

    servers = conf.get('local_server', conf['servers'])
    acl = get_acl(conf)

    zkh = zk_sync_init(servers, None, 10000)
    if 'auth' in conf:
        zookeeper.add_auth(zkh, 'digest', conf['auth'], None)

    node = args[0]

    if options.data is not None:
        if len(args) != 1:
            parser.error("Incorrect number of arguments");
        data = options.data
    elif options.yaml is not None:
        cont, _ = zookeeper.get(zkh, node)
        cont = yaml.safe_load(cont)
        for key, val in options.yaml:
            key_parts = key.split('.')
            cur = cont
            for key_part in key_parts[0:-1]:
                cur = cur[key_part]
            cur[key_parts[-1]] = yaml.safe_load(val)
        data = yaml.safe_dump(cont, explicit_start=True, default_flow_style=False)
    elif options.json is not None:
        cont, _ = zookeeper.get(zkh, node)
        cont = simplejson.loads(cont)
        for key, val in options.json:
            key_parts = key.split('.')
            cur = cont
            for key_part in key_parts[0:-1]:
                cur = cur[key_part]
            cur[key_parts[-1]] = yaml.safe_load(val)
        data = simplejson.dumps(cont, sort_keys=True, indent=4)
    else:
        if len(args) != 2:
            parser.error("Incorrect number of arguments");
        elif not os.path.exists(args[1]):
            parser.error("Data file %s does not exists" % args[1]);
        data = open(args[1]).read()

    node_exists = zookeeper.exists(zkh, node)
    if options.create and not node_exists:
        logging.info("create zk-node '%s' to '%s', acl: %s" % (node, data, acl))
        zookeeper.create(zkh, node, data, acl)
    else:
        if not node_exists:
            print "No such node: '%s', try --create flag" % node
            sys.exit(1)
        logging.info("set zk-node '%s' to '%s'" % (node, data))
        old_acl = zookeeper.get_acl(zkh, node)[1]
        acl = old_acl if old_acl else acl
        zookeeper.set(zkh, node, data)
        zookeeper.set_acl(zkh, node, -1, acl)
    logging.info("SUCCESS")

if __name__ == '__main__':
    main()
