#!/usr/bin/env python
# -*- encoding: utf-8 -*-

import argparse
import yaml
import os
import sys
import subprocess
import tempfile
import shutil
import re
import difflib
from kazoo.client import KazooClient

DELIMITER = "-" * 40
ZK_HOSTS = "ppcback01f.yandex.ru,ppcback01e.yandex.ru,ppcback01i.yandex.ru"
ZK_TIMEOUT = 10
ZK_RETRY = {"max_tries": 1, "delay": 1, "max_jitter": 1, "backoff": 1, "ignore_expire": False}
zkh = KazooClient(hosts=ZK_HOSTS, timeout=ZK_TIMEOUT, connection_retry=ZK_RETRY, command_retry=ZK_RETRY)


def with_delim(text):
    return "%s\n%s" % (DELIMITER, text)


def die(text):
    sys.stderr.write("%s\n" % (with_delim(text)))
    sys.exit(1)


def make_test(meta_conf, opts):
    print with_delim("RUNNING TESTS:")
    tests_passed = True
    selected_configs = set(opts.configs)

    for config_path in meta_conf:
        if selected_configs and config_path not in selected_configs:
            continue

        print "Checking: %s" % config_path
        config_content = ""
        with open(config_path, 'r') as fd:
            config_content = fd.read()

        test_scripts = [
            os.path.join(meta_conf[config_path]['tests_dir'], test_script)
            for test_script in os.listdir(meta_conf[config_path]['tests_dir'])
        ]
        for test_script in test_scripts:
            # пропускаем типичные временные файлы
            if re.match(r".*\.(swp)$", test_script):
                continue
            print "Running: %s" % test_script
            try:
                proc = subprocess.Popen(
                    test_script,
                    stdin=subprocess.PIPE
                )
                proc.communicate(config_content)
                if proc.returncode != 0:
                    tests_passed = False
                    print "ERROR: non-zero exit code %s" % proc.returncode
                else:
                    print "OK"
            except Exception as e:
                print "ERROR: something went wrong during test execution\nEXCEPTION: %s %s" % (type(e), e)
                tests_passed = False

    if tests_passed:
        print with_delim("OK")
    else:
        die("ERROR: some tests failed")


def check_working_copy():
    print with_delim("RUNNING WORKING COPY CHECK:")
    print "Checking whether the working copy is clean:"
    svn_info = subprocess.check_output(['svn', 'info'])
    svn_url = re.search(ur'^URL:\s*(.+)$', svn_info, re.M).group(1)

    tmp_dir = tempfile.mkdtemp(prefix='zk-sync-')
    try:
        print "Checkout %s to %s" % (svn_url, tmp_dir)
        svn_output = subprocess.check_output(['svn', 'co', svn_url, tmp_dir], stderr=subprocess.STDOUT)
        svn_diff = subprocess.check_output(['svn', 'diff', tmp_dir, './', '--summarize']).strip()
        if svn_diff:
            die("ERROR: working copy has uncommitted modifications\n%s" % svn_diff)

        print with_delim("OK")
    except Exception as e:
        die("ERROR: something went wrong during checking working copy\nEXCEPTION: %s %s" % (type(e), e))
    finally:
        shutil.rmtree(tmp_dir)


def sync(meta_conf, token_path, selected_configs, do):
    global zkh

    # после проверки рабочей копии можем безопасно делать up, конфликтов быть не должно
    print with_delim("RUNNING SVN UP:")
    subprocess.check_call(['svn', 'up'])

    print with_delim("RUNNING SYNC:")
    zkh.start()
    if do:
        try:
            zk_token = load_zookeeper_token(token_path)
        except Exception as e:
            print "Proceeding without auth: %s %s" % (type(e), e)
        else:
            zkh.add_auth("digest", zk_token)

    selected_configs = set(selected_configs)

    for config_path in meta_conf:
        try:
            if selected_configs and config_path not in selected_configs:
                continue
            
            zk_node = meta_conf[config_path]['zk_node']
            prev_data = ""
            if zkh.exists(zk_node):
                prev_data = zkh.get(zk_node)[0]
            else:
                print "Node %s doesn't exist, create it" % zk_node
                if do:
                    zkh.create(zk_node)

            cur_data = ""
            with open(config_path, 'r') as fd:
                cur_data = fd.read()

            if prev_data != cur_data:
                print "Updating %s from %s" % (zk_node, config_path)
                if do:
                    zkh.set(zk_node, cur_data)
                else:
                    print ''.join(difflib.unified_diff(
                        prev_data.splitlines(True), cur_data.splitlines(True), 'zk:' + zk_node, 'svn:' + config_path
                    ))
            else:
                print "No changes for %s" % zk_node
        except Exception as e:
            die("ERROR: can't sync %s and %s\nEXCEPTION: %s %s" % (zk_node, config_path, type(e), e))

    if do:
        print with_delim("OK: all configs are synced with ZK")
    else:
        print with_delim("DRY RUN: run again with --do")
    zkh.stop()
    return


def make_sync(meta_conf, opts):
    check_working_copy()
    make_test(meta_conf, opts)
    sync(meta_conf, opts.token_path, opts.configs, opts.do)


META_CONF_NAME = "meta_conf.yaml"
ACTIONS_DICT = {
    'test': {
        'def': make_test,
        'desc': 'запустить проверку конфигов в текущей рабочей копии',
    },
    'sync': {
        'def': make_sync,
        'desc': 'безопасно синхронизировать конфиги из svn в зукипер',
    },
}


def parse_options():
    parser = argparse.ArgumentParser(description="Синхронизация и проверка конфигов из зукипера")
    parser.add_argument(
        "action", type=str, choices=ACTIONS_DICT.keys(),
         help="; ".join("%s - %s" % (action, ACTIONS_DICT[action]['desc']) for action in ACTIONS_DICT),
    )
    parser.add_argument(
        'configs', nargs='*', metavar="config", type=str, default=[],
        help="конкретный конфиг для синхронизации из meta_conf.yaml (по умолчанию: все), можно указать несколько"
    )
    parser.add_argument(
        "-t", "--token", dest="token_path", type=str, default="~/.zk-sync-token",
        help="путь к файлу с токеном для зукипера (по умолчанию, ~/.zk-sync-token)"
    )
    parser.add_argument(
        "--do", dest="do", action='store_true', default=False,
        help="указать, чтобы изменения отправились в зукипер, без него только дифф"
    )
    opts, extra = parser.parse_known_args()

    if len(extra) > 0:
        die("There are unknown parameters: %s" % " ".join(extra))

    return opts


def load_meta_conf(conf_name=META_CONF_NAME):
    with open(conf_name, 'r') as fd:
        return yaml.load(fd)


def load_zookeeper_token(token_path):
    with open(os.path.expanduser(token_path), 'r') as fd:
        return fd.read().strip()


def run():
    opts = parse_options()
    meta_conf = load_meta_conf()
    ACTIONS_DICT[opts.action]['def'](meta_conf, opts)


if __name__ == '__main__':
    run()
