#!/usr/bin/perl

=head1 DESCRIPTION

Скрипт дергает zookeeper на указанном хосте (создает эфемерную ноду, читает ее и удаляет);
если все сработало ожидаемо -- отправляет в juggler событие OK.

Отдельно надо делать кронтаб, по которому этот скрипт будет запускаться с правильными параметрами. 

Для проверки локального zookeeper на стандартном порту 2181 достаточно указать только --juggler-service:

* * * * *   root  /usr/local/bin/zk-check.pl --juggler-service direct_infra.zookeeper.check > /dev/null 2>&1

Опции: 

 -h, --help             справка
 -H, --host <hostname>  хост, на котором работает zookeeper; по умолчанию localhost
 -p, --port <port>      порт, на котором работает zookeeper; по умолчанию 2181
 -n, --node <hodename>  имя ноды, которая будет использоваться для проверки; по умолчанию собирается из текущего хостнейма и суффикса _monitoring
 -j, --juggler-service  имя сервиса в juggler, на которыё отправлять событие

=cut

use warnings;
use strict;
use utf8;

use Getopt::Long;
use Net::ZooKeeper qw(:node_flags :acls);
use Net::Domain qw/hostfqdn/; 

use Yandex::JugglerQueue;


run() unless caller();

sub run
{
    my %O;

    Getopt::Long::Configure("no_ignore_case");
    GetOptions(
        'h|help' => sub {
            system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text-utf8");
            exit 0;
        },
        'H|host=s' => \$O{host},
        'p|port=s' => \$O{port},
        'n|node=s' => \$O{node_name},
        'j|juggler-service=s' => \$O{juggler_service},
    ) or die "can't parse options, stop\n";

    die "no --juggler-service given, stop\n" unless $O{juggler_service};

    $O{host} ||= 'localhost';
    $O{port} ||= 2181;

    my $node_name = $O{node_name};

    if (!$node_name){
        my $host = hostfqdn() =~ s/[^-a-z\.0-9]//gr; 
        $host =~ s/[\.-]+/_/g;
        $node_name = "/".$host."_monitoring";
    }

    my $zk_host = "$O{host}:$O{port}"; 

    my $zkh;
    for my $iter (1..5) {
        $zkh = Net::ZooKeeper->new($zk_host) or die "Error creating zkh object: $!";
        last if $zkh->exists("/");
    }

    my $check_time = time();

    $zkh->create($node_name, $check_time, flags => ZOO_EPHEMERAL, acl => ZOO_OPEN_ACL_UNSAFE) or die("unable to create node $node_name: " . $zkh->get_error());
    if (my $value = $zkh->get($node_name)) {
        if ($value == $check_time) {
            queue_juggler_event(service => $O{juggler_service}, status => "OK", description => "Sucessfully checked zookeeper \@ $zk_host");
            $zkh->delete($node_name);
            exit 0;
        } else {
            exit 1;
        }
    } else {
        exit 2;
    }
}

