# -*- coding:utf-8 -*-
"""
Скрипт для неинтерактивных обращений к zookeeper из командной строки
(пришел на замену direct-utils/zookeeper-tools/bin/direct-zkcli)

Общая справка:

    dt-zkcli --help

Справка по отдельной команде:

    dt-zkcli help ls

Команды:

    cat get
    ls
    rm delete
    tee set
    touch create
    vim
    getAcl
    setAcl
    stat

Токен:
    По умолчанию токен (user:password) берется из ~/.dt-zkcli.token
    Можно указать путь до файла с токеном через переменную окружения DIRECT_ZKCLI_TOKENFILE

Примеры:

    dt-zkcli -H directmod3.haze.yandex.net ls / -R
    dt-zkcli ls /playground
    dt-zkcli touch /playground/n1
    # в zookeeper храним только текстовые данные, поэтому на входе dt-zkcli tee должен быть валидный UTF-8
    echo "some data" | dt-zkcli tee /playground/n1
    dt-zkcli cat /playground/n1
    dt-zkcli rm /playground/n1

    dt-zkcli getAcl /some-node
    dt-zkcli setAcl /some-node world:anyone:r ip:127.0.0.1:rwdca digest:$(cat my_secret_node_token):rwcda
    dt-zkcli setAcl /some-node world:anyone:r ip:127.0.0.1:rwdca digest:user:password:rwcda

    dt-zkcli help setAcl
"""
import os
import sys
import argparse
import tempfile
import difflib
from kazoo.client import KazooClient
from kazoo.security import OPEN_ACL_UNSAFE, Permissions, ACL, Id, make_digest_acl_credential
from kazoo.exceptions import BadVersionError

DEFAULT_TOKEN_PATH = "~/.dt-zkcli.token"

ZK_TIMEOUT = 5
ZK_RETRY = {"max_tries": 3, "delay": 0.3, "backoff": 2, "ignore_expire": False}
zkh = None

PERM_DICT = {
    'r': Permissions.READ,
    'w': Permissions.WRITE,
    'c': Permissions.CREATE,
    'd': Permissions.DELETE,
    'a': Permissions.ADMIN
}


def cmd_cat(opts):
    """
    Читает указанные ключи и пишет контент на stdout

    dt-zkcli cat /playground/node1 /playground/node2
    """
    for node in opts.node:
        print(zkh.get(node)[0].decode('utf-8'))
    return


def cmd_ls(opts):
    """
    Выводит список из нод-детей

    -R -- рекурсивный листинг

    dt-zkcli ls /playground
    dt-zkcli ls / -R
    """
    if not opts.node:
        opts.node = ["/"]
    depth = 10000 if opts.recursive else 1
    subnodes = []

    for node in opts.node:
        get_children_recursive(node, depth, subnodes)

    print("\n".join(sorted(set(subnodes))))
    return


def get_children_recursive(node, depth, res):
    if depth <= 0:
        return

    depth -= 1
    try:
        subnodes = zkh.get_children(node)
    except Exception as e:
        return

    for subnode in subnodes:
        path = "/" + subnode if node == "/" else node + "/" + subnode
        res.append(path)
        if depth <= 0:
            continue
        get_children_recursive(path, depth, res)


def cmd_rm(opts):
    """
    Удаляет ноды

    dt-zkcli rm /playground/node1 /playground/node2
    """
    for node in opts.node:
        zkh.delete(node)


def cmd_tee(opts):
    """
    Читает stdin и пишет в указанную ноду

    echo "some text" | dt-zkcli tee /playground
    """
    if len(opts.node) != 1:
        sys.exit("tee expects exactly 1 parameter but %d given" % len(opts.node))

    cmd_touch(opts)
    zkh.set(opts.node[0], sys.stdin.buffer.read())


def cmd_touch(opts):
    """
    Cоздает указанные ноды, в том числе и рекурсивно
    Если ноды не было -- записывает пустую строку, если была -- не трогает

    dt-zkcli touch /playground/my_subnode/my_node
    """
    for node in opts.node:
        zkh.ensure_path(node, OPEN_ACL_UNSAFE)


def cmd_getAcl(opts):
    """
    Показывает acl-и на указанную ноду/ноды. В каждой строке: права, схема, id, нода.

    Права: (r)ead, (w)rite, (c)reate, (d)elete, (a)dmin

    > dt-zkcli -H ppcback01f.yandex.ru getAcl /direct/db-config.json /direct
    rwcda ip 127.0.0.1 /direct/db-config.json
    r---- world anyone /direct/db-config.json
    rwcda ip 127.0.0.1 /direct
    r---- world anyone /direct
    """
    for node in opts.node:
        for acl in zkh.get_acls(node)[0]:
            for perm in PERM_DICT:
                print(perm if PERM_DICT[perm] & acl.perms == PERM_DICT[perm] else "-", end='')
            print(" %s %s %s" % (acl.id.scheme, acl.id.id, node))


def cmd_setAcl(opts):
    """
    Устанавливает acl-и на указанную ноду.
    Указывать полный список прав (права заменяются, не добавляются).

    dt-zkcli setAcl /some-node world:anyone:r ip:127.0.0.1:rwdca digest:user:password:rwcda

    ОСТОРОЖНО! Можно нечаянно отнять у себя все права.

    Права: (r)ead, (w)rite, (c)reate, (d)elete, (a)dmin
    """
    if len(opts.node) < 2:
        sys.exit("setAcl expects at least 2 parameters but %d given" % len(opts.node))

    acls = []
    node = opts.node[0]
    admin_perms = False

    for acl_str in opts.node[1:]:
        scheme, tail = acl_str.split(':', 1)
        if scheme == 'digest':
            user, password, perms_str = tail.split(':')
            id = make_digest_acl_credential(user, password)
        else:
            id, perms_str = tail.split(':')

        perms = 0
        for perm in perms_str:
            if perm not in PERM_DICT:
                sys.exit("unknown permission '%s' in acl '%s', stop" % (perm, acl_str))
            admin_perms |= perm == 'a'
            perms |= PERM_DICT[perm]

        acl = ACL(perms, Id(scheme, id))
        acls.append(acl)

    if not admin_perms:
        print("DANGEROUS: no admin permission set, stop")
    zkh.set_acls(node, acls)


def cmd_vim(opts):
    """
    Интерактивное редактирование ноды с помощью vim.
    После выхода из редактора, показывает дифф и предлагает сохранить изменения в ноде, продолжить
    редактирование или отменить.
    Проверяется, что версия ноды актуальная.

    dt-zkcli vim /playground/node1
    """
    if len(opts.node) != 1:
        sys.exit("expects exactly 1 parameter but %d given" % len(opts.node))

    node = zkh.get(opts.node[0])
    tmp = tempfile.NamedTemporaryFile(delete=False)

    try:
        prev_data = node[0].decode("utf-8")
        tmp.write(node[0])
        tmp.close()

        while True:
            ret_code = os.system("vim " + tmp.name)
            if ret_code != 0:
                sys.exit("not saving new content because of non-zero exit code")

            with open(tmp.name, 'r') as fh:
                cur_data = fh.read()

            print(''.join(difflib.unified_diff(
                prev_data.splitlines(True), cur_data.splitlines(True), 'zk:%s' % opts.node[0], 'edited version'
            )))

            ans = input(
                "Check diff and choose action:\n(y) - save changes, " +
                "(e) - continue editing, any other key - exit without saving changes\n> "
            )

            if ans == 'y':
                break
            elif ans == 'e':
                print("\n##################################################\n")
                continue

            return

        with open(tmp.name, 'rb') as fh:
            new_data = fh.read()

        zkh.set(opts.node[0], new_data, node[1].version)
    except BadVersionError:
        print("Old version of node, run command again")
    finally:
        os.unlink(tmp.name)


def cmd_stat(opts):
    """
    Выполняет командy stat на заданные ноды

    dt-zkcli stat /playground/node1
    """
    for node in opts.node:
        print("%s:" % node)
        for key, value in zkh.get(node)[1]._asdict().items():
            print("  %s: %s" % (key, value))


def cmd_help(opts):
    """
    Выводит описание команд (без параметров выведет для всех)

    dt-zkcli help ls
    """
    if not opts.node:
        opts.node = list(CMDS.keys())

    for cmd in opts.node:
        if cmd in ALIASES_DICT:
            cmd = ALIASES_DICT[cmd]
        print("Help for command '%s'%s:" % (
            cmd,
            " (aliases: %s)" % ", ".join(CMDS[cmd]['aliases']) if CMDS.get(cmd, {}).get('aliases', []) else ""
        ))
        if cmd in CMDS:
            print(CMDS[cmd]['code'].__doc__ if CMDS[cmd]['code'].__doc__ else "-")
        else:
            print("error: can't find command")
        print("#" * 40)


CMDS = {
    'cat': {
        'code': cmd_cat,
        'aliases': ['get']
    },
    'ls': {
        'code': cmd_ls,
    },
    'rm': {
        'code': cmd_rm,
        'aliases': ['delete']
    },
    'tee': {
        'code': cmd_tee,
        'aliases': ['set']
    },
    'touch': {
        'code': cmd_touch,
        'aliases': ['create']
    },
    'getAcl': {
        'code': cmd_getAcl
    },
    'setAcl': {
        'code': cmd_setAcl
    },
    'vim': {
        'code': cmd_vim
    },
    'stat': {
        'code': cmd_stat
    },
    'help': {
        'code': cmd_help
    }
}

ALIASES_DICT = {aliase: cmd for cmd in CMDS for aliase in CMDS[cmd].get('aliases', [])}


def parse_options():
    parser = argparse.ArgumentParser(formatter_class=argparse.RawTextHelpFormatter, description=__doc__)
    parser.add_argument(
        'command',
        choices=list(CMDS.keys()) + list(ALIASES_DICT.keys()),
        help='команда для выполнения'
    )
    parser.add_argument('node', nargs='*', help='название ноды')
    parser.add_argument(
        '-H', '--hosts', type=str, default='127.0.0.1:2181',
        help="хосты с zookeeper'ом через запятую,\nнапример: 127.0.0.1:2181,127.0.0.1:2182,[::1]:2183"
    )
    parser.add_argument('-R', '--recursive', action='store_true', help='выполнить рекурсивно для всей ноды')
    parser.add_argument(
        '-t', '--token',
        type=str,
        help='путь до файла с токеном (по умолчанию: %s)' % DEFAULT_TOKEN_PATH
    )
    opts = parser.parse_args()
    opts.node = [os.path.normpath(x) for x in opts.node]

    if opts.command in ALIASES_DICT:
        opts.command = ALIASES_DICT[opts.command]

    if not opts.token:
        if os.environ.get('DIRECT_ZKCLI_TOKENFILE', ''):
            opts.token = os.environ['DIRECT_ZKCLI_TOKENFILE']
        else:
            opts.token = DEFAULT_TOKEN_PATH

    return opts


def main():
    global zkh

    opts = parse_options()

    auth_data = set([])
    try:
        with open(os.path.expanduser(opts.token), 'r') as fh:
            auth_data = [("digest", fh.read().strip())]
    except:
        pass

    # acl_local_rw = make_acl("ip", "127.0.0.1", all=True)
    # acl_world_ro = make_acl("world", "anyone", read=True
    zkh = KazooClient(
        hosts=opts.hosts,
        timeout=ZK_TIMEOUT,
        connection_retry=ZK_RETRY,
        command_retry=ZK_RETRY,
        auth_data=auth_data,
        # default_acl=[acl_local_rw, acl_world_ro]
    )
    zkh.start(timeout=ZK_TIMEOUT * len(opts.hosts.split(",")))
    CMDS[opts.command]['code'](opts)

    return


if __name__ == "__main__":
    main()
