# -*- coding: utf-8 -*-

from parser import NginxParser, JsrenderParser, MrateParser, TraceParser
from aggregator import JsrenderAggregator, MrateAggregator
import reader as draftsman_reader
from positioner import Positioner
import computings as draftsman_computings
import logging

from library.python.monlib.metric_registry import HistogramType

import datetime
import re
import yaml

VHOSTS_CONFIG = "/etc/yandex-direct/direct-vhosts.yaml"


class Log(object):
    config = None
    parser = None

    # скорее всего aggregator и computings не нужны, но для совместимости со старым draftsman оставил
    aggregator = None
    computings = None

    def __init__(self, config):
        self.config = config

    def read(self, registry):
        lognames = []
        for target_log in self.config["target_log"]:
            if target_log.get("logname_with_date", False):
                today = datetime.date.today()

                def logname_with_date(logname, logdate, joiner='.', dateformat='%Y%m%d'):
                    return joiner.join([logname, logdate.strftime(dateformat)])

                lognames.append(logname_with_date(target_log["path"], today - datetime.timedelta(1)))
                lognames.append(logname_with_date(target_log["path"], today))
            else:
                lognames.append(target_log["path"])

        positioner = Positioner(self.config["posfile"])
        for logname in lognames:
            reader = draftsman_reader.Reader(positioner=positioner, parser=self.parser)
            rd = reader.read(logname)

            for dtime, data in rd:
                self._create_sensors(registry, dtime, data)

        positioner.clean_posfile()

    def _create_sensors(self, registry, dtime, data):
        pass


class NginxLog(Log):
    parser = NginxParser()
    aggregator = None
    computings = None

    def __init__(self, config):
        with open('/etc/nginx/vhost.name', 'r') as fh:
            vhost = re.search(r'default\s+([^;]+)', fh.read()).group(1)

        with open(VHOSTS_CONFIG, 'r') as fh:
            config['target_log'] = [
                {
                    'path': yaml.load(fh, Loader=yaml.BaseLoader)['vhosts'][vhost]['nginx_access_log']
                }
            ]

        config['labels'] = {'vhost': vhost}
        super(NginxLog, self).__init__(config)

    def _create_sensors(self, registry, dtime, data):
        sensor_labels = self.config.get('labels', {})
        for item in data:
            sensor_labels['controller'] = item['controller']
            sensor_labels['response_code'] = item['response_code']
            sensor_labels['sensor'] = 'rps'

            registry.rate(sensor_labels).inc()

            sensor_labels['sensor'] = 'response_time'
            registry.histogram_rate(
                sensor_labels, HistogramType.Explicit, buckets=self.config["response_time_buckets"]
            ).collect(item['response_time'])


class TraceLog(Log):
    parser = TraceParser()

    def __init__(self, config):
        super(TraceLog, self).__init__(config)

    def _create_sensors(self, registry, dtime, data):
        sensor_names = ['all_ela', 'calls', 'obj_num']
        result_values = {}
        for row in data:
            for item in row.get('profiles', []):
                if (item['func'], item['tags']) not in result_values:
                    result_values[(item['func'], item['tags'])] = {sensor_name: 0.0 for sensor_name in sensor_names}
                for sensor_name in sensor_names:
                    try:
                        result_values[(item['func'], item['tags'])][sensor_name] += float(item[sensor_name])
                    except:
                        logging.exception("invalid value '%s' for sensor '%s', profile: '%s'" % (
                            item[sensor_name], sensor_name, row
                        ))

        for key, value in result_values.items():
            sensor_labels = {'func': key[0], 'tags': key[1]}
            for sensor in sensor_names:
                sensor_labels['sensor'] = sensor
                registry.gauge(sensor_labels).set(value[sensor])


class JsrenderLog(Log):
    parser = JsrenderParser()
    aggregator = JsrenderAggregator()
    computings = None

    def _create_sensors(self, registry, dtime, data):
        pass


class MrateLog(Log):
    parser = MrateParser()
    counts = ['errors', 'retries', 'timeouts', 'total']
    aggregator = MrateAggregator(counts=counts)

    def __init__(self, config):
        def cmp_func(key):
            return lambda _: _[key]

        funcs = [
            draftsman_computings.Computing(cnt_name, func=cmp_func(cnt_name))
            for cnt_name in self.counts
        ]

        percentiles = [
            draftsman_computings.CMP_PROC_80,
            draftsman_computings.CMP_PROC_90,
            draftsman_computings.CMP_PROC_95,
            draftsman_computings.CMP_PROC_98,
            draftsman_computings.CMP_PROC_99,
            draftsman_computings.CMP_PROC_100,
            draftsman_computings.CMP_AVG,
        ]
        for func in percentiles:
            def f(x, func=func):
                return func([{'response_time': t} for t in x['response_time']])

            funcs.append(draftsman_computings.Computing(func.name, func=f))

        self.computings = funcs

        super(MrateLog, self).__init__(config)

    def _create_sensors(self, registry, dtime, data):
        pass
