# -*- coding: utf-8 -*-

import datetime
import dateutil.parser
import json
import logging

from direct.direct_log.lib.trace_format import Trace


class Parser(object):
    '''
    Парсер строки лога.
    '''

    DO_SECOND = 6
    DO_MINUTE = 5
    DO_HOUR = 4
    DO_DAY = 3
    DO_MONTH = 2
    DO_YEAR = 1

    def __init__(self, datetime_order=DO_MINUTE):
        self.datetime_order = datetime_order

    def get_data(self, line):
        pass

    def get_group_datetime(self, dtime):
        '''
        Извлечение даты-времени с точностью, заданной при инициализации
        экземпляра класса.
        '''
        return datetime.datetime(*(dtime.timetuple()[:self.datetime_order] + (0,) * (6 - self.datetime_order)))

    def parse(self, line):
        data = self.get_data(line)
        return data


class NginxParser(Parser):
    def __init__(self, *args, **kwargs):
        self.aliases = kwargs.get('aliases')
        super(NginxParser, self).__init__(*args, **kwargs)

    def get_data(self, line):
        fields = line.split('\t')

        # Время ответа
        try:
            response_time = float(fields[9].split(':')[1])
        except ValueError:
            response_time = 0
        except:
            logging.exception("line: '%s'" % line)
            raise

        # Дато-временная метка
        dtime = dateutil.parser.parse(fields[2].strip('[]').replace(':', ' ', 1))

        try:
            data = {
                'controller': '4xx' if fields[4][0] == '4' else NginxParser.__get_controller_name(fields[10]),
                'datetime': dtime,
                'response_code': '%sxx' % fields[4][0],
                'response_time': response_time,
            }
        except IndexError:
            logging.exception("line: '%s'" % line)
            raise

        if self.aliases:
            data['controller'] = self.aliases.get(data['controller'], data['controller'])

        return data

    @staticmethod
    def __get_controller_name(value):
        if value == "-":
            return value

        controller_names = [
            cmd[1] for cmd in [pair.split(':') for pair in value.split(',')]
            if cmd and cmd[0] == 'cmd'
        ]
        controller_name = controller_names[0].strip() if controller_names else value.strip()

        bad_char_idx = len(controller_name)
        for i in xrange(len(controller_name)):
            if not controller_name[i].isalpha() and controller_name[i] not in ['_', '.', '/']:
                bad_char_idx = i
                break

        if bad_char_idx == len(controller_name):
            return controller_name

        last_char_idx = max(controller_name.rfind('.', 0, bad_char_idx), controller_name.rfind('/', 0, bad_char_idx))
        if last_char_idx == -1:
            return "bad.controller.name"

        return controller_name[:last_char_idx]


class TraceParser(Parser):
    def __init__(self):
        super(TraceParser, self).__init__()

    def get_data(self, line):
        data = Trace.decode(line)
        return {
            'datetime': data.chunk_end,
            'profiles': [{
                'all_ela': profile.all_ela,
                'calls': profile.calls,
                'obj_num': profile.obj_num,
                'func': profile.func,
                'tags': profile.tags
            } for profile in data.profiles]
        }


class JsrenderParser(Parser):
    def get_group_datetime(self, dtime):
        return dtime.replace(microsecond=0, second=0, minute=10 * int(dtime.minute / 10))

    def get_data(self, line):
        # 2014-09-15 00:00:46 showCamp/1581826870724351713 {"unload":45110}
        parts = line.split("\t")
        dt = datetime.datetime(*([int(i) for i in parts[0].split("-") + parts[1].split(":")]))
        data = json.loads(parts[3])
        data['datetime'] = dt
        data['controller'] = parts[2].split('/')[0]
        return data


class MrateParser(Parser):
    def get_data(self, line):
        parts = line.split("\t")
        dt = datetime.datetime(*([int(i) for i in parts[0].split("-") + parts[1].split(":")]))
        data = json.loads(parts[2])
        result = self.aggregate(data)
        result['datetime'] = dt
        return result

    def aggregate(self, items):
        errors = 0
        response_time = []
        retries = 0
        timeouts = 0
        total = 0
        for item in items:
            if item[0] == 'retry':
                retries += 1
            elif item[0] == 'timeout':
                timeouts += 1
            elif item[0] != 'success':
                # error, bad_response, eof
                errors += 1

            if len(item) != 1:
                response_time.append(item[2])
                total += 1

        return {
            'errors': errors,
            'response_time': response_time,
            'retries': retries,
            'timeouts': timeouts,
            'total': total,
        }
