# -*- coding: utf-8 -*-

import os
import datetime
import json
import dateutil.parser


class Positioner(object):
    '''
    Вспомогательный класс для работы с позициями, хранимыми между актами чтения
    файлов.
    '''

    def __init__(self, posfile):
        self.posfile = posfile

    def clean_posfile(self, lower_date=None):
        if lower_date is None:
            lower_date = datetime.date.today() - datetime.timedelta(1)

        cleaned_positions = dict([
            (k, v) for k, v in self.load_positions().items()
            if dateutil.parser.parse(v['datetime']).date() >= lower_date
        ])

        self.positions_to_file(cleaned_positions)

    def get_position_data(self, fdesc, dtime):
        return {
            'datetime': dtime.isoformat(),
            'inode': os.fstat(fdesc.fileno()).st_ino,
            'position': fdesc.tell(),
        }

    def load_positions(self):
        if os.path.exists(self.posfile):
            with open(self.posfile) as posfd:
                positions = json.load(posfd)
        else:
            positions = {}

        return positions

    def positions_to_file(self, pos_data):
        tmp_posfile = '%s.tmp' % self.posfile
        with open(tmp_posfile, 'w') as posfd:
            json.dump(pos_data, posfd)
        os.rename(tmp_posfile, self.posfile)

    def save_positions(self, fdesc, dtime):
        all_pos = self.load_positions()
        all_pos[fdesc.name] = self.get_position_data(fdesc, dtime)
        self.positions_to_file(all_pos)
