# -*- coding: utf-8 -*-

import argparse
import json
import sys
import logging
import logging.handlers

import flask

from library.python.monlib.metric_registry import MetricRegistry
from library.python.monlib.encoder import dumps

from draftsman.logs import NginxLog, JsrenderLog, MrateLog, TraceLog


CONTENT_TYPE_SPACK = 'application/x-solomon-spack'
CONTENT_TYPE_JSON = 'application/json'
CONFIG = {}
registry = MetricRegistry()


def parse_options():
    parser = argparse.ArgumentParser(
        description=u"Аналог solomon-agent, парсит логи из конфига и отдает сенсоры в соломон"
    )
    parser.add_argument(
        "-c", "--config", dest="config", type=str, required=True,
        help=u"путь до конфига"
    )

    parser.add_argument(
        "-p", "--port", dest="port", type=int, required=True,
        help=u"порт, на котором приложение будет слушать"
    )

    opts, extra = parser.parse_known_args()

    if len(extra) > 0:
        sys.exit("There are unknown parameters")

    return opts


def main():
    global CONFIG

    opts = parse_options()
    with open(opts.config) as fh:
        CONFIG = json.load(fh)

    for service in CONFIG:
        for prop in ["posfile", "type"]:
            if prop not in CONFIG[service]:
                sys.exit("ERROR: param '%s' is required for all services (service: %s)" % (prop, service))

    app = flask.Flask(__name__)

    sysloghandler = logging.handlers.SysLogHandler('/dev/log', facility="local0")
    sysloghandler.setFormatter(
        logging.Formatter('draftsman-solomon.log: (port ' + str(opts.port) + ') (%(levelname)s) %(message)s')
    )
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)
    logger.addHandler(sysloghandler)
    app.logger.setLevel(logging.DEBUG)
    app.logger.handlers = [sysloghandler]

    @app.route('/sensors')
    def sensors():
        global registry

        service = flask.request.args.get('service', '')
        if service not in CONFIG:
            return "service '%s' is not found in config" % service, 400

        if CONFIG[service]['type'] == "nginx":
            log = NginxLog(config=CONFIG[service])
        elif CONFIG[service]['type'] == "jsrender":
            log = JsrenderLog(config=CONFIG[service])
        elif CONFIG[service]['type'] == "mrate":
            log = MrateLog(config=CONFIG[service])
        elif CONFIG[service]['type'] == "trace":
            log = TraceLog(config=CONFIG[service])
        else:
            return "unsupported log type '%s'" % CONFIG[service]['type'], 400

        try:
            log.read(registry)
        except Exception as e:
            logging.exception("error during reading log")
            return "error during reading log: %s %s" % (e, type(e)), 500

        if flask.request.headers['accept'] == CONTENT_TYPE_SPACK:
            return flask.Response(dumps(registry), mimetype=CONTENT_TYPE_SPACK)

        return flask.Response(dumps(registry, format='json'), mimetype=CONTENT_TYPE_JSON)

    app.run(host='::', port=opts.port)


if __name__ == '__main__':
    main()
