package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"github.com/gofrs/flock"
	"io"
	"log"
	"net"
	"net/http"
	"os"
	"strings"
	"time"
)

/*
Демон для прослушивания определенного порта и логирования активности на нем.
Логи записываются в /var/log/dt-blackholle/current. Т.ж. получить список машин можно через запрос:
curl 'http://ppcgraphite02f.ppc.yandex.ru:8085/hosts?duration=5m' - активность за 5 минут
curl 'http://ppcgraphite02f.ppc.yandex.ru:8085/hosts?duration=24h' -активность за день
*/

const (
	PORT       = 2024
	LOCKFILE   = "/tmp/dt-blackholle.lock"
	DEBUG      = false
	LISTENPORT = 8085
)

var (
	port, lisport int
	lockfile      string
	debug         bool
)

type Connects map[string]time.Time

func (c Connects) Hostnames(interval string) (hostnames []string, err error) {
	duration, err := time.ParseDuration(interval)
	if err != nil {
		err = fmt.Errorf("error parse duration: %s", err)
		return
	}
	delta := time.Now().Add(-duration)
	for host, t := range c {
		if t.After(delta) {
			hostnames = append(hostnames, host)
		}
	}
	return
}

func Run(conn net.Conn, fd *os.File) {
	defer func() { _ = conn.Close() }()
	if _, err := io.Copy(fd, conn); err != nil {
		log.Printf("error copy: %s", err)
	}
}

func RunHTTP(connects Connects) {
	http.HandleFunc("/hosts", func(w http.ResponseWriter, r *http.Request) {
		_ = r.ParseForm()
		duration := r.Form.Get("duration")
		hostnames, err := connects.Hostnames(duration)
		if err != nil {
			w.Header().Set("Error", err.Error())
			http.Error(w, err.Error(), http.StatusInternalServerError)
		}
		if err := json.NewEncoder(w).Encode(hostnames); err != nil {
			log.Printf("error encode json for /hosts: %s", err)
		}
	})

	serv := &http.Server{
		Addr:         fmt.Sprintf(":%d", lisport),
		WriteTimeout: 30 * time.Second,
		ReadTimeout:  30 * time.Second,
	}
	if err := serv.ListenAndServe(); err != nil {
		log.Fatalf("error start http server %v: %s", serv, err)
	}
}

func main() {
	flag.IntVar(&port, "port", PORT, "listen port")
	flag.StringVar(&lockfile, "lock-file", LOCKFILE, "flock file")
	flag.BoolVar(&debug, "debug", DEBUG, "debug mode")
	flag.IntVar(&lisport, "listen-port", LISTENPORT, "listen port")
	flag.Parse()

	fileLock := flock.New(lockfile)

	locked, err := fileLock.TryLock()
	if err != nil {
		log.Fatalf("error trylock: %s", err)
	}

	if !locked {
		if debug {
			log.Printf("cannot get lock for %s", lockfile)
		}
		os.Exit(1)
	}

	defer func() { _ = fileLock.Unlock() }()

	connects := make(Connects)
	address := fmt.Sprintf(":%d", port)
	incoming, err := net.Listen("tcp", address)
	if err != nil {
		log.Fatalf("error listen %s: %s", address, err)
	}

	fd, err := os.OpenFile("/dev/null", os.O_RDWR, 0666)
	defer func() { _ = fd.Close() }()

	if err != nil {
		log.Fatalf("error open /dev/null: %s", err)
	}

	go RunHTTP(connects)

	go func(c Connects) {
		t := time.NewTicker(1 * time.Minute)
		for range t.C {
			h, err := c.Hostnames("1h")
			if err != nil {
				log.Fatalf("error ticker: %s", err)
			}
			log.Printf("last hour hosts: %s", strings.Join(h, ","))
		}
	}(connects)

	var name string
	for {
		conn, err := incoming.Accept()
		if err != nil {
			log.Printf("error incoming connect: %s", err)
			continue
		}
		cnaddr, ok := conn.RemoteAddr().(*net.TCPAddr)
		if !ok {
			name = conn.RemoteAddr().String()
		} else {
			ip := cnaddr.IP.String()
			hostnames, err := net.LookupAddr(ip)
			if err != nil {
				name = ip
			} else {
				name = strings.Join(hostnames, ",")
			}
		}
		connects[name] = time.Now()
		go Run(conn, fd)
	}
}
