package main

import (
	"bytes"
	"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"sort"
	"strings"

	s "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/support"
)

var err error
var f = fmt.Sprintf
var status s.SimpleStatus
var programName string = os.Args[0]

//Для парсинга массива в командной строке. Например для списка серверов или инстансов.
type listString []string

func (ls *listString) String() string {
	return fmt.Sprintf("%s", *ls)
}

func (ls *listString) Set(value string) error {
	*ls = append(*ls, strings.Split(value, ",")...)
	return nil
}

func (ls *listString) Strings() []string {
	return *ls
}

//Тут начинается магия, но будьте остороны с заклинаниями ;)
func run(command string, body []byte) []byte {
	var out []byte

	configPath := flag.String("config", "", "read config file")
	flag.Parse()

	configValues := s.ParseConfig(*configPath) //берем default
	request := fmt.Sprintf("http://%s:%d/%s", configValues.ListenHost, configValues.ListenPort, command)

	var response *http.Response
	if len(body) == 0 {
		if response, err = http.Get(request); err != nil {
			log.Fatalf("error connect server %s: %s", request, err)
		}
	} else {
		if response, err = http.Post(request, "application/json", bytes.NewReader(body)); err != nil {
			log.Fatal(f("error request server %s with body %s: %s", request, body, err))
		}
	}

	defer func() { _ = response.Body.Close() }()

	if out, err = ioutil.ReadAll(response.Body); err != nil {
		log.Fatal(f("error read body %s: %s", request, err))
	}
	return out
}

func main() {
	var complects s.Complects
	var verdict s.Verdict

	var nameComplect, groupName, stageName, command, finishStage string
	var postBody []byte
	var hostnames, nameInstances listString
	var enableReplicator, enableBinlogWriter bool
	flag.StringVar(&nameComplect, "complect", "last", "дата комплекта баз для восстановления (Example: last, 20190805)")
	flag.StringVar(&groupName, "group", "default", "группа для обновления баз (Example: dev7)")
	flag.Var(&hostnames, "hosts", "список серверов через запятую (Example: ppcdata1-02f.yandex.ru)")
	flag.Var(&nameInstances, "instances", "список имен БД через запятую(Example: ppcdata1, ppcdict)")
	flag.StringVar(&stageName, "newstage", "default", "имя stage состояния(example: ready_prepate_grants)")
	flag.StringVar(&command, "command", "", "команда на выполнение(example: help,list-complects)")
	flag.StringVar(&finishStage, "stage-end", "finish", "до какого stage включительно выполнять задачи")
	flag.BoolVar(&enableReplicator, "enable-replicator", false, "включить переливку БД реликатора YT")
	flag.BoolVar(&enableBinlogWriter, "enable-binlogwriter", false, "включить обновление БД binlogwriter")
	flag.Parse()
	if len(command) == 0 {
		command = flag.Arg(0)
	}

	if len(command) == 0 {
		log.Fatal("нехватает команды для выполнения")
	}

	switch command {
	case "list-complects": //список полных комплектов
		body := run(command, []byte{})
		if err = json.Unmarshal(body, &complects); err != nil {
			log.Fatalf("error unmarshal %s: %s\n", body, err)
		}
		for _, complectName := range complects.ListComplectNames() {
			backups := complects[complectName]
			sort.Sort(backups) //удобно когда все посортировано по датам
			fmt.Printf("для комплекта %s доступно:\n", complectName)
			for _, backup := range backups {
				fmt.Printf("\t(%s) %s размером %s\n", backup.GetInstance(),
					backup.NameBackup, backup.GetPrognosedSize().SizeHuman())
			}
		}
	case "plan-restore": //выводит приблизительный план восстановления баз. При реальном запуске может отличаться.
		request := s.NewPlanRequest(
			groupName,
			nameComplect,
			finishStage,
			nameInstances.Strings(),
			enableReplicator,
			enableBinlogWriter,
		)
		fmt.Println(request)
		if postBody, err = json.Marshal(request); err != nil {
			log.Fatal(f("ошибка в собзадинии запроса: %s\n", err))
		}
		body := run(command, postBody)
		if err = json.Unmarshal(body, &verdict); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		fmt.Printf("%+v", verdict)
		verdict.PrintPlan()
	case "list-servers": //Список машин, за которыми ведется наблюдение
		var hosts s.Hosts
		body := run(command, nil)
		if err = json.Unmarshal(body, &hosts); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		hosts.PrintHosts()
	case "gtid-status":
		var gtids s.GtidMysqlBinlog
		request := s.NewPlanRequest(
			groupName,
			nameComplect,
			finishStage,
			nameInstances.Strings(),
			false,
			false)
		if postBody, err = json.Marshal(request); err != nil {
			log.Fatal(f("ошибка в собзадинии запроса: %s\n", err))
		}
		body := run(command, postBody)
		fmt.Printf("%s\n", body)
		if err = json.Unmarshal(body, &gtids); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		gtids.Print()
	case "task-status", "monrun": //Список задач, назначенных на группы машин. Да, если групп много - будет печально это видеть.
		var groupsManager s.GroupsManager
		body := run("task-status", nil)
		if err = json.Unmarshal(body, &groupsManager); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		var groups s.Groups
		if groupName != "default" {
			for _, n := range strings.Split(groupName, ",") {
				groups = append(groups, s.Group(n))
			}
		} else {
			groups = groupsManager.GetAllGroups()
		}
		if command == "task-status" {
			groupsManager.PrintGroupManagerStatus(groups)
		}

		if command == "monrun" {
			monstat := groupsManager.MonitoringStatus(groups)
			if groups, ok := monstat["failed"]; ok {
				fmt.Printf("1; founded failed tasks; "+
					"use '%s -group %s task-status' for more information\n", programName, strings.Join(groups, ","))
				return
			}
			var aggrStatus bytes.Buffer
			for state, groups := range monstat {
				for _, group := range groups {
					aggrStatus.WriteString(f("%s: %s, ", group, state))
				}
			}
			fmt.Printf("0; %s\n", aggrStatus.String())
		}
	case "add-host": //добавить машину в группу. Данные заносятся в zookeeper.
		var hosts s.Hosts
		for _, host := range hostnames {
			hosts = append(hosts, s.NewHost(host, groupName))
		}
		if postBody, err = json.Marshal(hosts); err != nil {
			log.Fatal(f("ошибка в создании запроса: %s\n", err))
		}
		body := run(command, postBody)
		if err = json.Unmarshal(body, &status); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		fmt.Print(f("Статус выполнения: %s\n", status.Status))
	case "remove-host": //удалить машины из группы
		var hosts s.Hosts
		for _, host := range hostnames {
			hosts = append(hosts, s.NewHost(host, nil))
		}
		if postBody, err = json.Marshal(hosts); err != nil {
			log.Fatal(f("ошибка в создании запроса: %s\n", err))
		}
		body := run(command, postBody)
		if err = json.Unmarshal(body, &status); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		fmt.Print(f("Статус выполнения: %s\n", status.Status))
	case "restore-backup", "unpaused": //начать процедуру восстановления бекапа. Оно не остановится пока не выполнит все stage.
		request := s.NewPlanRequest(
			groupName,
			nameComplect,
			finishStage,
			nameInstances.Strings(),
			enableReplicator,
			enableBinlogWriter,
		)
		fmt.Println(request)
		if postBody, err = json.Marshal(request); err != nil {
			log.Fatal(f("ошибка в создании запроса: %s\n", err))
		}
		body := run(command, postBody)
		if err = json.Unmarshal(body, &status); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		fmt.Print(f("Статус выполнения: %s для %s\n", status.Status, nameInstances.Strings()))
	case "set-stage": //перевести группу в другой stage. Для перезапуска предыдущего stage надо добавить ready_<stage>
		stage := s.NewStage(groupName, stageName)
		if postBody, err = json.Marshal(stage); err != nil {
			log.Fatal(f("ошибка в создании запроса: %s", err))
		}
		body := run(command, postBody)
		if err = json.Unmarshal(body, &status); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		fmt.Print(f("Статус выполнения: %s\n", status.Status))
	case "restart-failed-tasks":
		body := run(command, nil)
		if err = json.Unmarshal(body, &status); err != nil {
			log.Fatal(f("error unmarshal %s: %s\n", body, err))
		}
		fmt.Print(f("Статус выполнения: %s\n", status.Status))

	default:
		fmt.Println(helptext)
		return
	}
}

var helptext string = fmt.Sprintf(`Запуск:
	%[1]s [--arg1 [--arg2]] command

Cписок команд(command):
	list-complects - получить список комплектов бекапа
	plan-restore - показать примерный план распаковки бекапа
	list-servers - список доступных серверов
	task-status - список выполняемых задач
	add-host - добавить сервер
	remove-host - убрать сервер
	restore-backup - восстановление бекапа
	set-stage - выставление нового stage
    gtid-status - получить вывод позиций GTID
	restart-failed-tasks - перезапуск сфейлившихся(всех групп) задач
	unpaused - снимает паузу с stage

Список аргументов(args):
	-group <group> - имя группы
	-hosts <host1, host2> - список имен серверов
	-newstage <stage> - имя stage(examples: ready_prepare_data, prepare_grants)
	-stage-end <stage> - указать завершающий stage, после которого будет пауза
	-instances <instance1, instance2> - отфильтровывает инстансы, с которомы будет производиться работа
	-command <cmd> - позволяет указывать команду в любом месте

Примеры:
	1. Посмотреть примерный план восстановления бекапов на dev7:
		%[1]s -group dev7 plan-restore
	2. Добавить хост в группу dev7:
		%[1]s -group dev7 -hosts ppcdiscord.yandex.ru,ppcback01f.yandex.ru add-hosts
	3. Удалить хост из группы dev7:
		%[1]s -group dev7 -hosts ppcdiscord.yandex.ru,ppcback01f.yandex.ru remove-hosts
	4. Запустить восстановление бекапа на dev7 с паузой после prepare-data:
		%[1]s -group dev7 -stage-end prepare-data restore-backup
	5. Запустить восстановление бекапа на dev7 с восстановкой до конца:
		%[1]s -group dev7 restore-backup
	6. Запустить восстановление бекапа на dev7 конкретных баз:
		%[1]s -group dev7 -instances ppcmonitor,ppcdict restore-backup
	7. Продолжить восстановление бекапа на dev7:
		%[1]s -group dev7 unpaused
	8. Перезапустить все задачи в состоянии FAILED:
		%[1]s restart-failed-tasks
	9. Выставление stage на dev7:
		%[1]s -group dev7 -newstage ready_prepare_grants set-stage`, programName)
