package main

import (
	"bytes"
	"context"
	"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"strconv"
	"strings"
	"time"

	_ "github.com/go-sql-driver/mysql"
	"github.com/jmoiron/sqlx"

	s "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/support"
	y "a.yandex-team.ru/direct/infra/dt-db-manager/pkg/yttransfer"
)

var (
	mysqlInsertQuery = "INSERT INTO recommendations_status(ClientID, type, cid, pid, bid, user_key_1, user_key_2, user_key_3," +
		"timestamp, status) VALUES (:client_id, :type, :cid, :pid, :bid, :user_key_1, :user_key_2, :user_key_3, :timestamp, :status)"
	mysqlDeleteQuery = "DELETE FROM recommendations_status WHERE type=:type AND cid=:cid AND pid=:pid AND bid=:bid " +
		"AND timestamp=:timestamp AND status=:status"

	ytCheckTable = "%s/combined/recommendations_status"
)

type Recomendation struct {
	ClientID  int    `db:"client_id" yson:"client_id"`
	Type      int    `db:"type" yson:"type"`
	Cid       int    `db:"cid" yson:"cid"`
	Pid       int    `db:"pid" yson:"pid"`
	Bid       int    `db:"bid" yson:"bid"`
	UserKey1  string `db:"user_key_1" yson:"user_key_1"`
	UserKey2  string `db:"user_key_2" yson:"user_key_2"`
	UserKey3  string `db:"user_key_3" yson:"user_key_3"`
	Timestamp int    `db:"timestamp" yson:"timestamp"`
	Status    string `db:"status" yson:"status"`
}

type Recomendations []Recomendation

func (r Recomendations) HasRecomendationByNumber(id int) bool {
	for _, recom := range r {
		if recom.ClientID == id {
			return true
		}
	}
	return false
}

type MysqlConnect struct {
	s.ReplicasFormat
	Number int
	Conn   *sqlx.DB
	Error  error
}

type MysqlConnects []MysqlConnect

func (mcs MysqlConnects) NumbersWithoutError() (result []string) {
	for _, instance := range mcs {
		if instance.Error == nil {
			result = append(result, strconv.Itoa(instance.Number))
		}
	}
	return
}

func (mcs MysqlConnects) AllErrors() error {
	var result bytes.Buffer
	for _, instance := range mcs {
		if instance.Error == nil {
			continue
		}
		msg := fmt.Sprintf("%s:%d: %s;", instance.Host, instance.Port, instance.Error)
		result.WriteString(msg)
	}
	return fmt.Errorf("%s", result.String())
}

func SelectYT(client y.YtConnect, query string, countRows int) (rows Recomendations) {
	cntx, cancel := context.WithTimeout(context.Background(), time.Minute*10)
	defer cancel()
	for try := 0; try < 3; try++ {
		time.Sleep(time.Second * 120) //ждем пока данные приедут в YT
		rows = []Recomendation{}
		log.Printf("run query %s\n", query)
		r, err := client.Client.SelectRows(cntx, query, nil)
		if err != nil {
			log.Printf("error query %s\n", query)
			continue
		}
		for r.Next() {
			var row Recomendation
			if err = r.Scan(&row); err != nil {
				log.Printf("error read row: %s\n", err)
				continue
			}
			fmt.Printf("%+v\n", row)
			rows = append(rows, row)
		}
		if err = r.Close(); err != nil {
			log.Printf("error close read query: %s\n", err)
		}
		if len(rows) != countRows {
			continue
		}
		if err = r.Close(); err != nil {
			log.Printf("error close row: %s\n", err)
		}
		break
	}
	return
}

var usage = "Программа для проверки работоспособности Replicator'a. На вход требуется файл с указанием инстансов и новых gtid позиций.\n" +
	"\tФормат файла:\n [{\"host\":\"ppctest-mysql01f.da.yandex.ru\",\"instance\":\"ppcdata10\",\"port\":3350,\n" +
	"\"group-name\":\"dev7\",\"replica-name\":\"\"},{\"host\":\"ppctest-mysql01f.da.yandex.ru\", ... }]" +
	" \tПримеры команд:\n" +
	" \treplicator-check -cluster-restore zeno -account direct -directory-restore //home/direct/test/mysql-sync/dev7" +
	" -yt-token /etc/direct-tokens/yt_robot-direct-yt-test -mysql-token /etc/direct-tokens/mysql_direct-test -mysql-user" +
	" direct-test -instances-file ./replicas-plan"

func PrintUsage() {
	fmt.Println(usage)
	flag.PrintDefaults()
}

//Пишем в mysql таблицу recommendations_status и вычитываем из YT;
func main() {
	destinationDir := flag.String("directory-restore", "//home/direct/test/mysql-sync/dev7/current", "путь куда привезти бекап")
	restoreCluster := flag.String("cluster-restore", "zeno", "кластер YT для восстановления бекапов")
	account := flag.String("account", "direct", "аккаунт в YT")
	ytToken := flag.String("yt-token", "/tmp/pass1", "токен для подключения к YT")
	mysqlToken := flag.String("mysql-token", "/tmp/pass2", "токен для подключения к mysql")
	mysqlUser := flag.String("mysql-user", "default", "user для подключения к mysql")
	mysqlDBName := flag.String("mysql-db", "ppc", "dbname для подключения к mysql")
	instancesFile := flag.String("instances-file", "", "файл с инстансами для подключения")
	flag.Usage = PrintUsage
	flag.Parse()

	ytreplicator := s.YtReplicator{
		SourceCluster:      *restoreCluster,
		SourceDir:          "",
		DestinationCluster: *restoreCluster,
		DestinationDir:     *destinationDir,
		Account:            *account,
		YTTokenFile:        *ytToken,
	}

	mydata, err := ioutil.ReadFile(*instancesFile)
	if err != nil {
		log.Fatalf("error read file %s: %s", *instancesFile, err)
	}
	var instances s.ReplicasFormats
	if err = json.Unmarshal(mydata, &instances); err != nil {
		log.Fatalf("error unmarshal %s: %s", mydata, err)
	}

	mysqlPassord, err := ioutil.ReadFile(*mysqlToken)
	if err != nil {
		log.Fatalf("error read %s: %s", *mysqlToken, err)
	}

	if len(instances) == 0 {
		log.Fatalf("empty list instance into file %s", *instancesFile)
	}

	var connects MysqlConnects
	for num, instance := range instances {
		if strings.Contains(instance.Instance, "sandbox") ||
			strings.Contains(instance.Instance, "ppcdict") ||
			strings.Contains(instance.Instance, "ppclog") ||
			strings.Contains(instance.Instance, "monitor") {
			continue
		}
		dbConnString := fmt.Sprintf("%s:%s@(%s:%d)/%s", *mysqlUser, mysqlPassord, instance.Host, instance.Port, *mysqlDBName)
		conn, err := sqlx.Connect("mysql", dbConnString)
		mc := MysqlConnect{instance,
			num,
			conn,
			err,
		}

		connects = append(connects, mc)
		if err != nil {
			log.Printf("error connect %s:%d: %s\n", instance.Host, instance.Port, err)
			continue
		}
		defer func() { _ = conn.Close() }()
	}

	fullPath := fmt.Sprintf(ytCheckTable, *destinationDir)
	numbersInstances := connects.NumbersWithoutError()

	if len(numbersInstances) == 0 {
		log.Fatalf("empty good connections: %s", connects.AllErrors())
	}

	ytconnector, err := s.NewGroupYtReplicator(ytreplicator)
	if err != nil {
		log.Fatal(err)
	}

	client := ytconnector.DestinationClient

	ytQuery := fmt.Sprintf("* from [%s] where client_id in (%s) and type=2 limit 100", fullPath, strings.Join(numbersInstances, ","))

	//удаляем старые записи из recommendations_status
	for _, instance := range connects {
		if instance.Error != nil {
			continue
		}
		dataDelete := Recomendation{
			Type:      2,
			Cid:       3,
			Pid:       4,
			Bid:       5,
			Timestamp: 1000000,
			Status:    "in_progress",
		}

		txd := instance.Conn.MustBegin()
		log.Printf("run delete query %s:%d\n", instance.Host, instance.Port)
		if _, err = txd.NamedExec(mysqlDeleteQuery, &dataDelete); err != nil {
			log.Printf("error exec delete query %s: %s\n", mysqlDeleteQuery, err)
		}
		if err := txd.Commit(); err != nil {
			log.Printf("error commit delete query: %s\n", err)
			continue
		}
		log.Printf("success delete query\n")
	}

	//вставляем новые записи в recommendations_status
	rows := SelectYT(client, ytQuery, 0)
	if len(rows) != 0 {
		log.Printf("not all yt rows has deleted: %+v\n", rows)
		log.Fatalf("replicator not working")
	}

	for _, instance := range connects {
		if instance.Error != nil {
			continue
		}

		dataInsert := Recomendation{
			instance.Number,
			2,
			3,
			4,
			5,
			"",
			"",
			"",
			1000000,
			"in_progress",
		}

		tx := instance.Conn.MustBegin()
		log.Printf("run insert query %s:%d\n", instance.Host, instance.Port)
		_, instance.Error = tx.NamedExec(mysqlInsertQuery, &dataInsert)
		if instance.Error != nil {
			log.Printf("error exec insert query: %s\n", instance.Error)
			continue
		}
		instance.Error = tx.Commit()
		if instance.Error != nil {
			log.Printf("error commit insert query %s: %s\n", mysqlInsertQuery, instance.Error)
			continue
		}
		log.Printf("success insert query\n")
	}

	rowsAfter := SelectYT(client, ytQuery, len(numbersInstances))

	//проверяем по всем успешным inserts, что они попали в БД YT
	var errmsg bytes.Buffer
	for _, instance := range connects {
		if instance.Error == nil && !rowsAfter.HasRecomendationByNumber(instance.Number) {
			msg := fmt.Sprintf("for instance %s:%d not found cid=%d", instance.Host, instance.Port, instance.Number)
			errmsg.WriteString(msg)
		}
	}

	if msg := connects.AllErrors(); msg != nil && len(msg.Error()) != 0 {
		log.Printf("warnings: %s", msg)
	}

	if errmsg.Len() == 0 {
		log.Printf("replicator working\n")
	} else {
		log.Fatalf("replicator not working: %s\n", errmsg.String())
	}
}
